// Copyright 1998 by Patrik Simons
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston,
// MA 02111-1307, USA.
//
// Patrik.Simons@hut.fi
#include <iostream.h>
#ifdef USEDOUBLE
#include <math.h>
#endif
#include "atomrule.h"
#include "program.h"

Atom::Atom (Program *p0)
  : p(p0)
{
  endHead = 0;
  endPos = 0;
  endNeg = 0;
  endUpper = 0;
  end = 0;
  headof = 0;
  head = 0;
  pos = 0;
  neg = 0;
  source = 0;
  posScore = 0;
  negScore = 0;
  name = 0;
  closure = false;
  Bpos = false;
  Bneg = false;
  visited = false;
  guess = false;
  isnant = false;
  dependsonTrue = false;
  dependsonFalse = false;
  computeTrue = false;
  computeFalse = false;
  backtracked = false;
  forced = false;
  in_etrue_queue = false;
  in_efalse_queue = false;
}

Atom::~Atom ()
{
  delete[] head;
  delete[] name;
}

void
Atom::etrue_push ()
{
  if (!in_etrue_queue && !in_efalse_queue)
    p->equeue.push (this);
  in_etrue_queue = true;
  if (Bneg || in_efalse_queue)
    p->conflict = true;
}

void
Atom::efalse_push ()
{
  if (!in_etrue_queue && !in_efalse_queue)
    p->equeue.push (this);
  in_efalse_queue = true;
  if (Bpos || in_etrue_queue)
    p->conflict = true;
}

inline void
Atom::queue_push ()
{
  p->queue.push (this);
}

void 
Atom::backchainTrue ()
{
  Follows *f;
  // Find the only active rule.
  for (f = head; f != endHead; f++)
    if (!f->r->isInactive ())
      break;
  if (!f->r->isFired ())
    f->r->backChainTrue ();
}

//
// Note that since backchain depends on the truth values of the
// atoms in the body of the rules, we must set Bpos first.
// To keep everything safe we also handle inactivation before 
// we fire any rules.
//
void
Atom::setBTrue ()
{
  Follows *f;
  Bpos = true;
  for (f = neg; f != endNeg; f++)
    f->r->inactivate (f->a);
  for (f = pos; f != endPos; f++)
    f->r->mightFire (f->a);
  // This atom could have been forced.
  if (headof == 1)
    backchainTrue ();
}
 
void
Atom::setBFalse ()
{
  Follows *f;
  Bneg = true;
  for (f = pos; f != endPos; f++)
    f->r->inactivate (f->a);
  for (f = neg; f != endNeg; f++)
    f->r->mightFire (f->a);
  closure = false;
  source = 0;
  // This atom could have been forced.
  if (headof)          // There are active rules.
    backchainFalse (); // Might backchain already backchained rules
		       // in mightFireFalse.
}

void
Atom::setTrue ()
{
  for (Follows *f = pos; f != endPos; f++)
    f->r->mightFireInit (f->a);
  closure = true;
}

void
Atom::backtrackFromBTrue ()
{
  Follows *f;
  for (f = pos; f != endPos; f++)
    f->r->backtrackFromActive (f->a);
  for (f = neg; f != endNeg; f++)
    f->r->backtrackFromInactive (f->a);
  Bpos = false;
  closure = true;
}

void
Atom::backtrackFromBFalse ()
{
  Follows *f;
  for (f = neg; f != endNeg; f++)
    f->r->backtrackFromActive (f->a);
  for (f = pos; f != endPos; f++)
    f->r->backtrackFromInactive (f->a);
  Bneg = false;
  closure = true;
}

void 
Atom::backchainFalse ()
{
  for (Follows *f = head; f != endHead; f++)
    if (!f->r->isInactive () && !f->r->isFired ())
      f->r->backChainFalse ();
}

void
Atom::reduce_head ()
{
  Follows t;
  Follows *g = head;
  for (Follows *f = head; f != endHead; f++)
    {
      f->r->swapping (f, g);
      g->r->swapping (g, f);
      t = *g;
      *g = *f;
      *f = t;
      if (!g->r->isInactive ())
	g++;
    }
  endHead = g;
}

void
Atom::reduce_pbody (bool strong)
{
  Follows t;
  Follows *g = pos;
  for (Follows *f = pos; f != endPos; f++)
    {
      f->r->swapping (f, g);
      g->r->swapping (g, f);
      t = *g;
      *g = *f;
      *f = t;
      if ((strong == false || Bpos == false) &&
	  Bneg == false && !g->r->isInactive ())
	g++;
    }
  endPos = g;
  endUpper = g;
}

void
Atom::reduce_nbody (bool strong)
{
  Follows t;
  Follows *g = neg;
  for (Follows *f = neg; f != endNeg; f++)
    {
      f->r->swapping (f, g);
      g->r->swapping (g, f);
      t = *g;
      *g = *f;
      *f = t;
      if (Bpos == false && Bneg == false && !g->r->isInactive ())
	g++;
    }
  endNeg = g;
}

void
Atom::reduce (bool strongly)
{
  reduce_head ();
  reduce_pbody (strongly);
  reduce_nbody (strongly);
}

void
Atom::unreduce ()
{
  endHead = pos;
  endPos = neg;
  endUpper = neg;
  endNeg = end;
}

const char *
Atom::atom_name ()
{
  if (name)
    return name;
  else
    return "#noname#";
}

void
Atom::visit ()
{
  p->queue.push (this);
  visited = true;
}

BasicRule::BasicRule ()
{
  head = 0;
  lit = 0;
  upper = 0;
  nbody = 0;
  pbody = 0;
  nend = 0;
  pend = 0;
  end = 0;
  inactive = 0;
  visited = false;
  type = BASICRULE;
}

BasicRule::~BasicRule ()
{
  delete[] nbody;  // pbody is allocated after nbody
}

bool
BasicRule::isInactive ()
{
  return inactive;
}

bool
BasicRule::isUpperActive ()
{
  return upper == 0 && !inactive;
}

bool
BasicRule::isFired ()
{
  return lit == 0;
}

void
BasicRule::inactivate (Atom **a)
{
  if (a >= pbody && (*a)->closure)
    upper++;
  inactive++;
  if (inactive == 1)
    {
      Atom *b = head;
      b->headof--;
      if (b->Bneg == false)
	{
	  if (b->headof && (b->source == 0 || b->source == this))
	    {
	      b->source = 0;
	      b->queue_push ();
	    }
	  if (b->headof == 0)
	    b->efalse_push ();
	  else if (b->headof == 1 && b->Bpos && *a != b)
	    b->backchainTrue ();
	}
    }
}

void
BasicRule::fireInit ()
{
  if (lit == 0)
    {
      head->etrue_push ();
      head->queue_push ();
      head->source = this;
    }
  else if (upper == 0)
    {
      head->queue_push ();
      if (head->source == 0)
	head->source = this;
    }
}

void
BasicRule::mightFireInit (Atom **)
{
  upper--;
  if (upper == 0 && !inactive)
    {
      head->queue_push ();
      if (head->source == 0)
	head->source = this;
    }
}

void
BasicRule::unInit ()
{
  upper = pend-pbody;
}

void
BasicRule::mightFire (Atom **)
{
  lit--;
  if (lit == 0 && head->Bpos == false)
    head->etrue_push ();
  else if (lit == 1 && !inactive && head->Bneg == true)
    backChainFalse ();
}

void
BasicRule::backChainTrue ()
{
  for (Atom **b = nbody; b != nend; b++)
    if ((*b)->Bneg == false)
      (*b)->efalse_push ();
  for (Atom **b = pbody; b != pend; b++)
    if ((*b)->Bpos == false)
      (*b)->etrue_push ();
}

void
BasicRule::backChainFalse ()
{
  if (lit == 1)
    {
      for (Atom **b = nbody; b != nend; b++)
	if ((*b)->Bneg == false)
	  {
	    (*b)->etrue_push ();
	    return;
	  }
      for (Atom **b = pbody; b != pend; b++)
	if ((*b)->Bpos == false)
	  {
	    (*b)->efalse_push ();
	    return;
	  }
    }
}

void
BasicRule::backtrackFromActive (Atom **)
{
  lit++;
}

void
BasicRule::backtrackFromInactive (Atom **a)
{
  if (a >= pbody)
    upper--;
  inactive--;
  if (inactive == 0)
    head->headof++;
}

void
BasicRule::propagateFalse (Atom **)
{
  upper++;
  if (upper == 1 && !inactive && head->closure != false &&
      (head->source == 0 || head->source == this))
    {
      head->source = 0;
      head->queue_push ();
    }
}

void
BasicRule::propagateTrue (Atom **)
{
  upper--;
  if (upper == 0 && !inactive && head->closure == false)
    {
      if (head->source == 0)
	head->source = this;
      head->queue_push ();
    }
}

void
BasicRule::backtrackUpper (Atom **)
{
  upper--;
}

void
BasicRule::search (Atom *)
{
  if (!head->visited && head->Bneg == false)
    head->visit ();
  for (Atom **a = nbody; a != nend; a++)
    if (!(*a)->visited && (*a)->Bneg == false)
      (*a)->visit ();
  for (Atom **a = pbody; a != pend; a++)
    if (!(*a)->visited && (*a)->Bneg == false)
      (*a)->visit ();
  visited = true;
}

void
BasicRule::reduce (bool strongly)
{
  Atom **a;
  Atom **b;
  Atom *t;
  b = nbody;
  for (a = nbody; a != nend; a++)
    {
      t = *b;
      *b = *a;
      *a = t;
      if ((*b)->Bpos == false && (*b)->Bneg == false)
	b++;
    }
  nend = b;
  b = pbody;
  for (a = pbody; a != pend; a++)
    {
      t = *b;
      *b = *a;
      *a = t;
      if ((strongly == false || (*b)->Bpos == false)
	  && (*b)->Bneg == false)
	b++;
    }
  pend = b;
}

void
BasicRule::unreduce ()
{
  nend = pbody;
  pend = end;
}

void
BasicRule::setup ()
{
  head->head->r = this;
  head->head->a = &head;
  head->head++;
  Atom **a;
  for (a = nbody; a != nend; a++)
    {
      (*a)->negScore--;
      (*a)->neg[(*a)->negScore].r = this;
      (*a)->neg[(*a)->negScore].a = a;
    }
  for (a = pbody; a != pend; a++)
    {
      (*a)->posScore--;
      (*a)->pos[(*a)->posScore].r = this;
      (*a)->pos[(*a)->posScore].a = a;
    }
}

void
BasicRule::print ()
{
  cout << head->atom_name ();
  if (nbody)
    cout << " :- ";
  Atom **a;
  int comma = 0;
  for (a = pbody; a != pend; a++)
    {
      if (comma)
	cout << ", ";
      cout << (*a)->atom_name ();
      comma = 1;
    }
  for (a = nbody; a != nend; a++)
    {
      if (comma)
	cout << ", ";
      cout << "not " << (*a)->atom_name ();
      comma = 1;
    }
  cout << '.' << endl;
}


ConstraintRule::ConstraintRule ()
{
  head = 0;
  lit = 0;
  upper = 0;
  nbody = 0;
  pbody = 0;
  nend = 0;
  pend = 0;
  end = 0;
  inactive = 0;
  visited = false;
  type = CONSTRAINTRULE;
}

ConstraintRule::~ConstraintRule ()
{
  delete[] nbody;
}

bool
ConstraintRule::isInactive ()
{
  return inactive > 0;
}

bool
ConstraintRule::isUpperActive ()
{
  return upper <= 0 && inactive <= 0;
}

bool
ConstraintRule::isFired ()
{
  return lit <= 0 && inactive <= 0;
}

void
ConstraintRule::inactivate (Atom **a)
{
  if (!(a >= pbody && (*a)->closure == false))
    upper++;
  inactive++;
  if (inactive == 1)
    {
      Atom *b = head;
      b->headof--;
      if (b->Bneg == false)
	{
	  if (b->headof && (b->source == 0 || b->source == this))
	    {
	      b->source = 0;
	      b->queue_push ();
	    }
	  if (b->headof == 0)
	    b->efalse_push ();
	  else if (b->headof == 1 && b->Bpos && *a != b)
	    b->backchainTrue ();
	}
    }
}

void
ConstraintRule::fireInit ()
{
  if (lit <= 0)
    {
      head->etrue_push ();
      head->queue_push ();
      head->source = this;
    }
  else if (upper <= 0)
    {
      head->queue_push ();
      if (head->source == 0)
	head->source = this;
    }
}

void
ConstraintRule::mightFireInit (Atom **)
{
  upper--;
  if (upper == 0 && !isInactive ())
    {
      head->queue_push ();
      if (head->source == 0)
	head->source = this;
    }
}

void
ConstraintRule::unInit ()
{
  upper = lit - (nend-nbody);
}

void
ConstraintRule::mightFire (Atom **)
{
  lit--;
  if (lit == 0 && head->Bpos == false)
    head->etrue_push ();
  else if (lit == 1 && inactive <= 0 && head->Bneg == true)
    backChainFalse ();
}

void
ConstraintRule::backChainTrue ()
{
  if (inactive == 0)
    {
      for (Atom **b = nbody; b != nend; b++)
	if ((*b)->Bneg == false && (*b)->Bpos == false)
	  (*b)->efalse_push ();
      for (Atom **b = pbody; b != pend; b++)
	if ((*b)->Bneg == false && (*b)->Bpos == false)
	  (*b)->etrue_push ();
    }
}

void
ConstraintRule::backChainFalse ()
{
  if (lit == 1)
    {
      for (Atom **b = nbody; b != nend; b++)
	if ((*b)->Bneg == false && (*b)->Bpos == false)
	  (*b)->etrue_push ();
      for (Atom **b = pbody; b != pend; b++)
	if ((*b)->Bneg == false && (*b)->Bpos == false)
	  (*b)->efalse_push ();
    }
}

void
ConstraintRule::backtrackFromActive (Atom **)
{
  lit++;
}

void
ConstraintRule::backtrackFromInactive (Atom **a)
{
  upper--;
  inactive--;
  if (inactive == 0)
    head->headof++;
}

void
ConstraintRule::propagateFalse (Atom **)
{
  upper++;
  if (upper == 1 && !isInactive () && head->closure != false &&
      (head->source == 0 || head->source == this))
    {
      head->source = 0;
      head->queue_push ();
    }
}

void
ConstraintRule::propagateTrue (Atom **)
{
  upper--;
  if (upper == 0 && !isInactive () && head->closure == false)
    {
      if (head->source == 0)
	head->source = this;
      head->queue_push ();
    }
}

void
ConstraintRule::backtrackUpper (Atom **)
{
  upper--;
}

void
ConstraintRule::search (Atom *)
{
  if (!head->visited && head->Bneg == false)
    head->visit ();
  for (Atom **a = nbody; a != nend; a++)
    if (!(*a)->visited && (*a)->Bneg == false)
      (*a)->visit ();
  for (Atom **a = pbody; a != pend; a++)
    if (!(*a)->visited && (*a)->Bneg == false)
      (*a)->visit ();
  visited = true;
}

void
ConstraintRule::reduce (bool strongly)
{
  Atom **a;
  Atom **b;
  Atom *t;
  b = nbody;
  for (a = nbody; a != nend; a++)
    {
      t = *b;
      *b = *a;
      *a = t;
      if ((*b)->Bpos == false && (*b)->Bneg == false)
	b++;
    }
  nend = b;
  b = pbody;
  for (a = pbody; a != pend; a++)
    {
      t = *b;
      *b = *a;
      *a = t;
      if ((strongly == false || (*b)->Bpos == false) &&
	  (*b)->Bneg == false)
	b++;
    }
  pend = b;
}

void
ConstraintRule::unreduce ()
{
  nend = pbody;
  pend = end;
}

void
ConstraintRule::setup ()
{
  head->head->r = this;
  head->head->a = &head;
  head->head++;
  Atom **a;
  for (a = nbody; a != nend; a++)
    {
      (*a)->negScore--;
      (*a)->neg[(*a)->negScore].r = this;
      (*a)->neg[(*a)->negScore].a = a;
    }
  for (a = pbody; a != pend; a++)
    {
      (*a)->posScore--;
      (*a)->pos[(*a)->posScore].r = this;
      (*a)->pos[(*a)->posScore].a = a;
    }
}

void
ConstraintRule::print ()
{
  cout << head->atom_name () << " :- ";
  Atom **a;
  int comma = 0;
  if (pbody != pend)
    {
      long atleast = lit;
      for (a = pbody; a != pend; a++)
	if ((*a)->Bpos)
	  atleast++;
      for (a = nbody; a != nend; a++)
	if ((*a)->Bneg)
	  atleast++;
      cout << atleast << " { ";
      for (a = pbody; a != pend; a++)
	{
	  if (comma)
	    cout << ", ";
	  cout << (*a)->atom_name ();
	  comma = 1;
	}
      for (a = nbody; a != nend; a++)
	{
	  if (comma)
	    cout << ", ";
	  cout << "not " << (*a)->atom_name ();
	  comma = 1;
	}
      cout << " }";
    }
  cout << '.' << endl;
}

GenerateRule::GenerateRule ()
{
  head = 0;
  hend = 0;
  pbody = 0;
  pend = 0;
  end = 0;
  neg = 0;
  pos = 0;
  upper = 0;
  inactivePos = 0;
  inactiveNeg = 0;
  visited = false;
  type = GENERATERULE;
}

GenerateRule::~GenerateRule ()
{
  delete[] head;
}

bool
GenerateRule::isInactive ()
{
  return inactivePos || inactiveNeg > 0;
}

bool
GenerateRule::isUpperActive ()
{
  return upper == 0 && !isInactive ();
}

bool
GenerateRule::isFired ()
{
  return pos == 0 && neg <= 0;
}

void
GenerateRule::inactivate (Atom **a)
{
  if (a >= pbody)
    {
      if ((*a)->closure)
	upper++;
      inactivePos++;
      if (!(inactivePos == 1 && inactiveNeg <= 0))
	return;
    }
  else
    {
      inactiveNeg++;
      if (!(inactiveNeg == 1 && inactivePos == 0))
	return;
    }
  for (Atom **h = head; h != hend; h++)
    {
      Atom *b = *h;
      b->headof--;
      if (b->Bneg == false)
	{
	  if (b->headof && (b->source == 0 || b->source == this))
	    {
	      b->source = 0;
	      b->queue_push ();
	    }
	  if (b->headof == 0)
	    b->efalse_push ();
	  else if (b->headof == 1 && b->Bpos && *a != b)
	    b->backchainTrue ();
	}
    }
}

void
GenerateRule::fireInit ()
{
  if (pos == 0 && neg <= 0)
    for (Atom **h = head; h != hend; h++)
      {
	(*h)->etrue_push ();
	(*h)->queue_push ();
	(*h)->source = this;
      }
  else if (upper == 0)
    for (Atom **h = head; h != hend; h++)
      {
	(*h)->queue_push ();
	if ((*h)->source == 0)
	  (*h)->source = this;
      }
}

void
GenerateRule::mightFireInit (Atom **)
{
  upper--;
  if (upper == 0 && !isInactive ())
    for (Atom **h = head; h != hend; h++)
      {
	(*h)->queue_push ();
	if ((*h)->source == 0)
	  (*h)->source = this;
      }
}

void
GenerateRule::unInit ()
{
  upper = pos;
}

void
GenerateRule::mightFire (Atom **a)
{
  if (a >= pbody)
    pos--;
  else
    neg--;
  if (pos == 0 && neg == 0)
    {
      for (Atom **h = head; h != hend; h++)
	if ((*h)->Bneg == false && (*h)->Bpos == false)
	  (*h)->etrue_push ();
    }
  else if (pos == 1 && neg < 0 && !isInactive ())
    backChainFalse ();
}

void
GenerateRule::backChainTrue ()
{
  if (inactiveNeg != 0)
    return;
  Atom **b;
  if (neg)
    for (b = head; b != hend; b++)
      if ((*b)->Bneg == false && (*b)->Bpos == false)
	(*b)->efalse_push ();
  if (pos)
    for (b = pbody; b != pend; b++)
      if ((*b)->Bpos == false)
	(*b)->etrue_push ();
}

void
GenerateRule::backChainFalse ()
{
  if (pos == 1 && neg < 0)
    for (Atom **b = pbody; b != pend; b++)
      if ((*b)->Bpos == false)
	{
	  (*b)->efalse_push ();
	  break;
	}
}

void
GenerateRule::backtrackFromActive (Atom **a)
{
  if (a >= pbody)
    pos++;
  else
    neg++;
}

void
GenerateRule::backtrackFromInactive (Atom **a)
{
  if (a >= pbody)
    {
      upper--;
      inactivePos--;
      if (inactivePos == 0 && inactiveNeg <= 0)
	for (Atom **h = head; h != hend; h++)
	  (*h)->headof++;
    }
  else
    {
      inactiveNeg--;
      if (inactiveNeg == 0 && inactivePos == 0)
	for (Atom **h = head; h != hend; h++)
	  (*h)->headof++;
    }
}

void
GenerateRule::propagateFalse (Atom **)
{
  upper++;
  if (upper == 1 && !isInactive ())
    {
      if (inactiveNeg < 0)
	{
	  for (Atom **h = head; h != hend; h++)
	    if ((*h)->closure != false &&
		((*h)->source == 0 || (*h)->source == this))
	      {
		(*h)->source = 0;
		(*h)->queue_push ();
	      }
	}
      else
	for (Atom **h = head; h != hend; h++)
	  if ((*h)->closure != false && (*h)->Bpos &&
	      ((*h)->source == 0 || (*h)->source == this))
	    {
	      (*h)->source = 0;
	      (*h)->queue_push ();
	    }
    }
}

void
GenerateRule::propagateTrue (Atom **)
{
  upper--;
  if (upper == 0 && !isInactive ())
    {
      if (inactiveNeg < 0)
	{
	  for (Atom **h = head; h != hend; h++)
	    if ((*h)->closure == false)
	      {
		if ((*h)->source == 0)
		  (*h)->source = this;
		(*h)->queue_push ();
	      }
	}
      else
	for (Atom **h = head; h != hend; h++)
	  if ((*h)->closure == false && (*h)->Bpos)
	    {
	      if ((*h)->source == 0)
		(*h)->source = this;
	      (*h)->queue_push ();
	    }
    }
}

void
GenerateRule::backtrackUpper (Atom **)
{
  upper--;
}

void
GenerateRule::search (Atom *)
{
  for (Atom **a = head; a != hend; a++)
    if (!(*a)->visited && (*a)->Bneg == false)
      (*a)->visit ();
  for (Atom **a = pbody; a != pend; a++)
    if (!(*a)->visited && (*a)->Bneg == false)
      (*a)->visit ();
  visited = true;
}

void
GenerateRule::reduce (bool strongly)
{
  Atom **a;
  Atom **b;
  Atom *t;
  b = head;
  for (a = head; a != hend; a++)
    {
      t = *b;
      *b = *a;
      *a = t;
      if ((*b)->Bpos == false && (*b)->Bneg == false)
	b++;
    }
  hend = b;
  b = pbody;
  for (a = pbody; a != pend; a++)
    {
      t = *b;
      *b = *a;
      *a = t;
      if ((strongly == false || (*b)->Bpos == false) &&
	  (*b)->Bneg == false)
	b++;
    }
  pend = b;
}

void
GenerateRule::unreduce ()
{
  hend = pbody;
  pend = end;
}

void
GenerateRule::setup ()
{
  Atom **a;
  for (a = head; a != hend; a++)
    {
      (*a)->head->r = this;
      (*a)->head->a = a;
      (*a)->head++;
    }
  for (a = head; a != hend; a++)
    {
      (*a)->negScore--;
      (*a)->neg[(*a)->negScore].r = this;
      (*a)->neg[(*a)->negScore].a = a;
    }
  for (a = pbody; a != pend; a++)
    {
      (*a)->posScore--;
      (*a)->pos[(*a)->posScore].r = this;
      (*a)->pos[(*a)->posScore].a = a;
    }
}

void
GenerateRule::print ()
{
  Atom **a;
  int comma = 0;
  long atleast = inactiveNeg;
  for (a = head; a != hend; a++)
    if ((*a)->Bpos == true)
      atleast--;
  cout << -atleast << " {";
  for (a = head; a != hend; a++)
    {
      if (comma)
	cout << ", ";
      cout << (*a)->atom_name ();
      comma = 1;
    }
  cout << '}';
  if (pbody != pend)
    cout << " :- ";
  comma = 0;
  for (a = pbody; a != pend; a++)
    {
      if (comma)
	cout << ", ";
      cout << (*a)->atom_name ();
      comma = 1;
    }
  cout << '.' << endl;
}

ChoiceRule::ChoiceRule ()
{
  head = 0;
  hend = 0;
  lit = 0;
  upper = 0;
  nbody = 0;
  pbody = 0;
  nend = 0;
  pend = 0;
  end = 0;
  inactive = 0;
  visited = false;
  type = CHOICERULE;
}

ChoiceRule::~ChoiceRule ()
{
  delete[] head;
}

bool
ChoiceRule::isInactive ()
{
  return inactive;
}

bool
ChoiceRule::isUpperActive ()
{
  return upper == 0 && !inactive;
}

bool
ChoiceRule::isFired ()
{
  return lit == 0;
}

void
ChoiceRule::inactivate (Atom **a)
{
  if (a >= pbody && (*a)->closure)
    upper++;
  inactive++;
  if (inactive == 1)
    {
      for (Atom **h = head; h != hend; h++)
	{
	  Atom *b = *h;
	  b->headof--;
	  if (b->Bneg == false)
	    {
	      if (b->headof && (b->source == 0 || b->source == this))
		{
		  b->source = 0;
		  b->queue_push ();
		}
	      if (b->headof == 0)
		b->efalse_push ();
	      else if (b->headof == 1 && b->Bpos && *a != b)
		b->backchainTrue ();
	    }
	}
    }
}

void
ChoiceRule::fireInit ()
{
  if (lit == 0 || upper == 0)
    for (Atom **h = head; h != hend; h++)
      {
	(*h)->queue_push ();
	if ((*h)->source == 0)
	  (*h)->source = this;
      }
}

void
ChoiceRule::mightFireInit (Atom **)
{
  upper--;
  if (upper == 0 && !inactive)
    for (Atom **h = head; h != hend; h++)
      {
	(*h)->queue_push ();
	if ((*h)->source == 0)
	  (*h)->source = this;
      }
}

void
ChoiceRule::unInit ()
{
  upper = pend-pbody;
}

void
ChoiceRule::mightFire (Atom **)
{
  lit--;
}

void
ChoiceRule::backChainTrue ()
{
  if (lit)
    {
      for (Atom **b = nbody; b != nend; b++)
	if ((*b)->Bneg == false)
	  (*b)->efalse_push ();
      for (Atom **b = pbody; b != pend; b++)
	if ((*b)->Bpos == false)
	  (*b)->etrue_push ();
    }
}

void
ChoiceRule::backChainFalse ()
{
}

void
ChoiceRule::backtrackFromActive (Atom **)
{
  lit++;
}

void
ChoiceRule::backtrackFromInactive (Atom **a)
{
  if (a >= pbody)
    upper--;
  inactive--;
  if (inactive == 0)
    for (Atom **h = head; h != hend; h++)
      (*h)->headof++;
}

void
ChoiceRule::propagateFalse (Atom **)
{
  upper++;
  if (upper == 1 && !inactive)
    for (Atom **h = head; h != hend; h++)
      if ((*h)->closure != false &&
	  ((*h)->source == 0 || (*h)->source == this))
	{
	  (*h)->source = 0;
	  (*h)->queue_push ();
	}
}

void
ChoiceRule::propagateTrue (Atom **)
{
  upper--;
  if (upper == 0 && !inactive)
    for (Atom **h = head; h != hend; h++)
      if ((*h)->closure == false)
	{
	  if ((*h)->source == 0)
	    (*h)->source = this;
	  (*h)->queue_push ();
	}
}

void
ChoiceRule::backtrackUpper (Atom **)
{
  upper--;
}

void
ChoiceRule::search (Atom *)
{
  for (Atom **h = head; h != hend; h++)
    if (!(*h)->visited && (*h)->Bneg == false)
      (*h)->visit ();
  for (Atom **a = nbody; a != nend; a++)
    if (!(*a)->visited && (*a)->Bneg == false)
      (*a)->visit ();
  for (Atom **a = pbody; a != pend; a++)
    if (!(*a)->visited && (*a)->Bneg == false)
      (*a)->visit ();
  visited = true;
}

void
ChoiceRule::reduce (bool strongly)
{
  Atom **a;
  Atom **b;
  Atom *t;
  b = head;
  for (a = head; a != hend; a++)
    {
      t = *b;
      *b = *a;
      *a = t;
      if ((strongly == false || (*b)->Bpos == false)
	  && (*b)->Bneg == false)
	b++;
    }
  hend = b;
  b = nbody;
  for (a = nbody; a != nend; a++)
    {
      t = *b;
      *b = *a;
      *a = t;
      if ((*b)->Bpos == false && (*b)->Bneg == false)
	b++;
    }
  nend = b;
  b = pbody;
  for (a = pbody; a != pend; a++)
    {
      t = *b;
      *b = *a;
      *a = t;
      if ((strongly == false || (*b)->Bpos == false)
	  && (*b)->Bneg == false)
	b++;
    }
  pend = b;
}

void
ChoiceRule::unreduce ()
{
  hend = nbody;
  nend = pbody;
  pend = end;
}

void
ChoiceRule::setup ()
{
  Atom **a;
  for (a = head; a != hend; a++)
    {
      (*a)->head->r = this;
      (*a)->head->a = a;
      (*a)->head++;
    }
  for (a = nbody; a != nend; a++)
    {
      (*a)->negScore--;
      (*a)->neg[(*a)->negScore].r = this;
      (*a)->neg[(*a)->negScore].a = a;
    }
  for (a = pbody; a != pend; a++)
    {
      (*a)->posScore--;
      (*a)->pos[(*a)->posScore].r = this;
      (*a)->pos[(*a)->posScore].a = a;
    }
}

void
ChoiceRule::print ()
{
  Atom **a;
  bool comma = false;
  cout << "{ ";
  for (a = head; a != hend; a++)
    {
      if (comma)
	cout << ", ";
      cout << (*a)->atom_name ();
      comma = true;
    }
  cout << " }";
  comma = false;
  if (pbody != pend || nbody != nend)
    cout << " :- ";
  for (a = pbody; a != pend; a++)
    {
      if (comma)
	cout << ", ";
      cout << (*a)->atom_name ();
      comma = true;
    }
  for (a = nbody; a != nend; a++)
    {
      if (comma)
	cout << ", ";
      cout << "not " << (*a)->atom_name ();
      comma = true;
    }
  cout << '.' << endl;
}


WeightRule::WeightRule ()
{
  head = 0;
  bend = 0;
  body = 0;
  end = 0;
  reverse = 0;
  weight = 0;
  positive = 0;
  minweight = 0;
  maxweight = 0;
  upper_min = 0;
  atleast = 0;
  max = 0;
  min = 0;
  visited = false;
  type = WEIGHTRULE;
}

WeightRule::~WeightRule ()
{
  delete[] body;
  delete[] weight;
  delete[] positive;
  delete[] reverse;
}

bool
WeightRule::isInactive ()
{
  return maxweight < atleast;
}

bool
WeightRule::isUpperActive ()
{
  return upper_min >= atleast;
}

bool
WeightRule::isFired ()
{
  return minweight >= atleast;
}

void
WeightRule::change (Atom *a)
{
  if (isInactive ())
    {
      Atom *b = head;
      b->headof--;
      if (b->Bneg == false)
	{
	  if (b->headof && (b->source == 0 || b->source == this))
	    {
	      b->source = 0;
	      b->queue_push ();
	    }
	  if (b->headof == 0)
	    b->efalse_push ();
	  else if (b->headof == 1 && b->Bpos && a != b)
	    b->backchainTrue ();
	}
    }
  else if (isFired ())
    {
      if (head->Bpos == false)
	head->etrue_push ();
    }
  else if (head->Bneg)
    backChainFalse ();
  else if (head->Bpos && head->headof == 1)
    backChainTrue ();
}

void
WeightRule::inactivate (Atom **a)
{
  if (positive[a-body] && (*a)->closure)
    upper_min -= weight[a-body];
  bool inactive = isInactive ();
  bool frd = isFired ();
  maxweight -= weight[a-body];
  if (!frd && !inactive)
    change (*a);
}

void
WeightRule::fireInit ()
{
  if (isFired ())
    {
      head->etrue_push ();
      head->queue_push ();
      head->source = this;
    }
  else if (isUpperActive () && !isInactive ())
    {
      head->queue_push ();
      if (head->source == 0)
	head->source = this;
    }
}

void
WeightRule::mightFireInit (Atom **a)
{
  bool active = isUpperActive ();
  upper_min += weight[a-body];
  if (!active && isUpperActive () && !isInactive ())
    {
      head->queue_push ();
      if (head->source == 0)
	head->source = this;
    }
}

void
WeightRule::unInit ()
{
  upper_min = 0;
  for (Atom **a = body; a != bend; a++)
    if (!positive[a-body])
      upper_min += weight[a-body];
}

void
WeightRule::mightFire (Atom **a)
{
  bool inactive = isInactive ();
  bool frd = isFired ();
  minweight += weight[a-body];
  if (!frd && !inactive)
    change (*a);
}

void
WeightRule::backChainTrue ()
{
  Atom **a = max;
  while (a != bend)
    {
      if ((*a)->Bpos || (*a)->Bneg)
	a++;
      else if (maxweight - weight[a-body] < atleast)
	{
	  if (positive[a-body])
	    (*a)->etrue_push ();
	  else
	    (*a)->efalse_push ();
	  a++;
	}
      else
	break;
    }
  if (max != bend && ((*max)->Bpos || (*max)->Bneg))
    max = a;
}

void
WeightRule::backChainFalse ()
{
  Atom **a = min;
  while (a != bend)
    {
      if ((*a)->Bpos || (*a)->Bneg)
	a++;
      else if (minweight + weight[a-body] >= atleast)
	{
	  if (positive[a-body])
	    (*a)->efalse_push ();
	  else
	    (*a)->etrue_push ();
	  a++;
	}
      else
	break;
    }
  if (min != bend && ((*min)->Bpos || (*min)->Bneg))
    min = a;
}

void
WeightRule::backtrack (Atom **a)
{
  if (max > a)
    max = a;
  if (min > a)
    min = a;
}

void
WeightRule::backtrackFromActive (Atom **a)
{
  minweight -= weight[a-body];
  backtrack (a);
}

void
WeightRule::backtrackFromInactive (Atom **a)
{
  bool inactive = isInactive ();
  Weight w = weight[a-body];
  maxweight += w;
  if (positive[a-body])
    upper_min += w;
  if (inactive && !isInactive ())
    head->headof++;
  backtrack (a);
}

void
WeightRule::propagateFalse (Atom **a)
{
  bool active = isUpperActive ();
  bool inactive = isInactive ();
  upper_min -= weight[a-body];
  if (!inactive && active && !isUpperActive () &&
      head->closure != false &&
      (head->source == 0 || head->source == this))
    {
      head->source = 0;
      head->queue_push ();
    }
}

void
WeightRule::propagateTrue (Atom **a)
{
  bool active = isUpperActive ();
  bool inactive = isInactive ();
  upper_min += weight[a-body];
  if (!inactive && !active && isUpperActive () &&
      head->closure == false)
    {
      if (head->source == 0)
	head->source = this;
      head->queue_push ();
    }
}

void
WeightRule::backtrackUpper (Atom **a)
{
  upper_min += weight[a-body];
}

void
WeightRule::search (Atom *a)
{
  if (a->Bneg || (a->Bpos && a != head))
    return;
  if (!head->visited && head->Bneg == false)
    head->visit ();
  for (Atom **b = body; b != bend; b++)
    if (!(*b)->visited && (*b)->Bneg == false)
      (*b)->visit ();
  visited = true;
}

void
WeightRule::reduce (bool strongly)
{
  Atom **b = body;
  for (Atom **a = body; a != bend; a++)
    {
      swap (b-body, a-body);
      if (positive[b-body])
	{
	  if ((strongly == false || (*b)->Bpos == false) &&
	      (*b)->Bneg == false)
	    b++;
	}
      else if ((*b)->Bpos == false && (*b)->Bneg == false)
	b++;
    }
  bend = b;
  backtrack (body);
}

void
WeightRule::unreduce ()
{
  bend = end;
  backtrack (body);
  sort ();
}

void
WeightRule::initWeight (Weight w)
{
  maxweight += w;
}

void
WeightRule::swap (long a, long b)
{
  Weight tw = weight[a];
  weight[a] = weight[b];
  weight[b] = tw;
  Atom *ta = body[a];
  body[a] = body[b];
  body[b] = ta;
  bool tb = positive[a];
  positive[a] = positive[b];
  positive[b] = tb;
  if (reverse[a])
    reverse[a]->a = body+b;
  if (reverse[b])
  reverse[b]->a = body+a;
}

bool
WeightRule::smaller (long a, long b)
{
  if (weight[a] < weight[b])
    return true;
  else
    return false;
}

void
WeightRule::heap (long k, long e)
{
  long a = 2*k+1;
  while (a < e)
    {
      if (smaller (a+1, a))
	a++;
      if (smaller (a, k))
	swap (a, k);
      else
	break;
      k = a;
      a = 2*k+1;
    }
  if (a == e && smaller (a, k))
    swap (a, k);
}

void
WeightRule::sort ()
{
  long i;
  long e = bend-body-1;
  for (i = (e-1)/2; i >= 0; i--)
    heap (i, e);
  i = e;
  while (i > 0)
    {
      swap (i, 0);
      i--;
      heap (0, i);
    }
}

void
WeightRule::swapping (Follows *f, Follows *g)
{
  if (f->a != &head)
    reverse[f->a - body] = g;
}

void
WeightRule::setup ()
{
  head->head->r = this;
  head->head->a = &head;
  head->head++;
  Atom **a;
  Weight *w = weight;
  sort ();
  for (a = body; a != bend; a++)
    if (positive[a-body])
      {
	(*a)->posScore--;
	(*a)->pos[(*a)->posScore].r = this;
	(*a)->pos[(*a)->posScore].a = a;
	reverse[a-body] = (*a)->pos + (*a)->posScore;
	initWeight (*w++);
      }
    else
      {
	(*a)->negScore--;
	(*a)->neg[(*a)->negScore].r = this;
	(*a)->neg[(*a)->negScore].a = a;
	reverse[a-body] = (*a)->neg + (*a)->negScore;
	upper_min += *w;
	initWeight (*w++);
      }
}

void
WeightRule::print ()
{
  cout << head->atom_name () << " :- { ";
  int comma = 0;
  for (Atom **a = body; a != bend; a++)
    if (positive[a-body])
      {
	if (comma)
	  cout << ", ";
	cout << (*a)->atom_name () << " = " << weight[a-body];
	comma = 1;
      }
    else
      {
	if (comma)
	  cout << ", ";
	cout << "not " << (*a)->atom_name () << " = "
	     << weight[a-body];
	comma = 1;
      }
  cout << "} >=" << atleast << '.' << endl;
}


OptimizeRule::OptimizeRule ()
{
  nbody = 0;
  pbody = 0;
  nend = 0;
  pend = 0;
  end = 0;
  weight = 0;
  minweight = 0;
  maxweight = 0;
  maxoptimum = 0;
  minoptimum = 0;
  visited = false;
  type = OPTIMIZERULE;
  maximize = false;
  next = 0;
}

OptimizeRule::~OptimizeRule ()
{
  delete[] nbody;
  delete[] weight;
}

void
OptimizeRule::setOptimum ()
{
  minoptimum = minweight;
  maxoptimum = maxweight;
}

bool
OptimizeRule::isInactive ()
{
  return false;  // Reduce(Strongly|Weakly) needs this
}

bool
OptimizeRule::isUpperActive ()
{
  return false;
}

bool
OptimizeRule::isFired ()
{
  return true;
}

void
OptimizeRule::inactivate (Atom **a)
{
  Weight w = weight[a-nbody];
  if (w >= 0)
    maxweight -= w;
  else
    minweight -= w;
}

void
OptimizeRule::fireInit ()
{
}

void
OptimizeRule::mightFireInit (Atom **)
{
}

void
OptimizeRule::unInit ()
{
}

void
OptimizeRule::mightFire (Atom **a)
{
  Weight w = weight[a-nbody];
  if (w < 0)
    maxweight += w;
  else
    minweight += w;
}

void
OptimizeRule::backChainTrue ()
{
}

void
OptimizeRule::backChainFalse ()
{
}

void
OptimizeRule::backtrackFromActive (Atom **a)
{
  Weight w = weight[a-nbody];
  if (w < 0)
    maxweight -= w;
  else
    minweight -= w;
}

void
OptimizeRule::backtrackFromInactive (Atom **a)
{
  Weight w = weight[a-nbody];
  if (w >= 0)
    maxweight += w;
  else
    minweight += w;
}

void
OptimizeRule::propagateFalse (Atom **)
{
}

void
OptimizeRule::propagateTrue (Atom **)
{
}

void
OptimizeRule::backtrackUpper (Atom **)
{
}

void
OptimizeRule::search (Atom *a)
{
  if (a->Bneg || a->Bpos)
    return;
  Atom **b;
  for (b = nbody; b != nend; b++)
    if (!(*b)->visited && (*b)->Bneg == false && (*b)->Bpos == false)
      (*b)->visit ();
  for (b = pbody; b != pend; b++)
    if (!(*b)->visited && (*b)->Bneg == false && (*b)->Bpos == false)
      (*b)->visit ();
  visited = true;
}

void
OptimizeRule::reduce (bool)
{
  Atom **a;
  Atom **b;
  Atom *t;
  b = nbody;
  for (a = nbody; a != nend; a++)
    {
      t = *b;
      *b = *a;
      *a = t;
      if ((*b)->Bpos == false && (*b)->Bneg == false)
	b++;
    }
  nend = b;
  b = pbody;
  for (a = pbody; a != pend; a++)
    {
      t = *b;
      *b = *a;
      *a = t;
      if ((*b)->Bpos == false && (*b)->Bneg == false)
	b++;
    }
  pend = b;
}

void
OptimizeRule::unreduce ()
{
  nend = pbody;
  pend = end;
}

void
OptimizeRule::initWeight (Weight w)
{
  if (w > 0)
    maxweight += w;
  else
    minweight += w;
}

void
OptimizeRule::setup ()
{
  Atom **a;
  Weight *w = weight;
  for (a = nbody; a != nend; a++)
    {
      (*a)->negScore--;
      (*a)->neg[(*a)->negScore].r = this;
      (*a)->neg[(*a)->negScore].a = a;
      initWeight (*w++);
    }
  for (a = pbody; a != pend; a++)
    {
      (*a)->posScore--;
      (*a)->pos[(*a)->posScore].r = this;
      (*a)->pos[(*a)->posScore].a = a;
      initWeight (*w++);
    }
}

void
OptimizeRule::print ()
{
  if (maximize)
    cout << "maximize { ";
  else
    cout << "minimize { ";
  Atom **a;
  int comma = 0;
  for (a = pbody; a != pend; a++)
    {
      if (comma)
	cout << ", ";
      cout << (*a)->atom_name () << " = " << weight[a-nbody];
      comma = 1;
    }
  for (a = nbody; a != nend; a++)
    {
      if (comma)
	cout << ", ";
      cout << "not " << (*a)->atom_name () << " = " << weight[a-nbody];
      comma = 1;
    }
  cout << " }" << endl;
}
