// Copyright 1998 by Patrik Simons
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston,
// MA 02111-1307, USA.
//
// Patrik.Simons@hut.fi
#include <iostream.h>
#include <stdlib.h>
#include "stable.h"

Stable::Stable ()
  : reader (&smodels.program)
{
  wellfounded = false;
  lookahead = true;
  backjumping = false;
  sloppy_heuristic = false;
  iterative_deepening = false;
  max_conflicts = 0;
  max_tries = 1;
  seed = 1;
}

Stable::~Stable ()
{
}

int
Stable::read (istream &f)
{
  timer.start ();
  int r = reader.read (f);
  if (r == 0)
    {
      smodels.init ();
      smodels.max_models = reader.models;
    }
  return r;
}

int
Stable::tries ()
{
  srand (1);   // Reset
  srand (seed);
  smodels.setup ();
  int r = smodels.conflict ();
  long tries;
  for (tries = 0; tries < max_tries && !r; tries++)
    {
      smodels.shuffle ();
      r = smodels.smodels (lookahead, backjumping, max_conflicts);
    }
  if (tries == 0 || r == -1)
    return 0;
  else
    return 1;
}

// Iterative deepening may print the same model many times!
int
Stable::iterate ()
{
  int r = -1;
  for (long i = 2; r == -1; i++)
    {
      cout << "Iterative depth " << i << endl;
      smodels.iterative_depth = i;
      r = smodels.iterate (lookahead, backjumping);
      smodels.unwind_to_setup ();
    }
  return r;
}

void
Stable::calculate ()
{
  int r;
  smodels.sloppy_heuristic = sloppy_heuristic;
  if (wellfounded)
    r = smodels.wellfounded ();
  else if (max_conflicts == 0)
    {
      if (seed != 1)
	{
	  srand (1);   // Reset
	  srand (seed);
	  smodels.shuffle ();
	}
      if (iterative_deepening)
	r = iterate ();
      else
	r = smodels.smodels (lookahead, backjumping);
    }
  else
    r = tries ();
  if (r)
    cout << "True" << endl;
  else
    cout << "False" << endl;
  timer.stop ();
  cout << "Duration: " << timer.print () << endl;
  cout << "Number of choice points: " <<
    smodels.number_of_choice_points << endl;
  cout << "Number of wrong choices: " <<
    smodels.number_of_wrong_choices << endl;
  cout << "Number of atoms: " << smodels.program.number_of_atoms <<
    endl;
  cout << "Number of rules: " << smodels.program.number_of_rules <<
    endl;
  if (lookahead)
    {
      cout << "Number of picked atoms: " <<
	smodels.number_of_picked_atoms << endl;
      cout << "Number of forced atoms: " <<
	smodels.number_of_forced_atoms << endl;
      cout << "Number of truth assignments: " <<
	smodels.number_of_assignments << endl;
    }
  if (backjumping)
    cout << "Number of backjumps: " << smodels.number_of_backjumps <<
      endl;
}
