// Copyright 1998 by Patrik Simons
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston,
// MA 02111-1307, USA.
//
// Patrik.Simons@hut.fi
#include <iostream.h>
#ifdef USEDOUBLE
#include <math.h>
#endif
#include "atomrule.h"
#include "program.h"

Atom::Atom (Program *p0)
  : p(p0)
{
  endHead = 0;
  endPos = 0;
  endNeg = 0;
  endPos2 = 0;
  end = 0;
  headof = 0;
  head = 0;
  pos = 0;
  neg = 0;
  source = 0;
  posScore = 0;
  negScore = 0;
  name = 0;
  closure = false;
  Bpos = false;
  Bneg = false;
  visited = false;
  guess = false;
  isnant = false;
  dependsonTrue = false;
  dependsonFalse = false;
  computeTrue = false;
  computeFalse = false;
  backtracked = false;
  forced = false;
  in_etrue_queue = false;
  in_efalse_queue = false;
}

Atom::~Atom ()
{
  delete[] head;
  delete[] name;
}

void
Atom::etrue_push ()
{
  if (!in_etrue_queue && !in_efalse_queue)
    p->equeue.push (this);
  in_etrue_queue = true;
  if (Bneg || in_efalse_queue)
    p->conflict = true;
}

void
Atom::efalse_push ()
{
  if (!in_etrue_queue && !in_efalse_queue)
    p->equeue.push (this);
  in_efalse_queue = true;
  if (Bpos || in_etrue_queue)
    p->conflict = true;
}

inline void
Atom::queue_push ()
{
  p->queue.push (this);
}

void 
Atom::backchainTrue ()
{
  Follows *f;
  // Find the only active rule.
  for (f = head; f != endHead; f++)
    if (!f->r->isInactive ())
      break;
  f->r->backChainTrue ();
}

//
// Note that since backchain depends on the truth values of the
// atoms in the body of the rules, we must set Bpos first.
// To keep everything safe we also handle inactivation before 
// we fire any rules.
//
void
Atom::setBTrue ()
{
  Follows *f;
  Bpos = true;
  for (f = neg; f != endNeg; f++)
    f->r->inactivateInFalse (f->a);
  for (f = pos; f != endPos; f++)
    f->r->mightFireTrue (f->a);
  // This atom could have been forced.
  if (headof == 1)
    backchainTrue ();
}
 
void
Atom::setBFalse ()
{
  Follows *f;
  Bneg = true;
  for (f = pos; f != endPos; f++)
    f->r->inactivateInTrue (f->a);
  for (f = neg; f != endNeg; f++)
    f->r->mightFireFalse (f->a);
  closure = false;
  source = 0;
  // This atom could have been forced.
  if (headof)          // There are active rules.
    backchainFalse (); // Might backchain already backchained rules
		       // in mightFireFalse.
}

void
Atom::setTrue ()
{
  for (Follows *f = pos; f != endPos; f++)
    f->r->mightFireInitTrue (f->a);
  closure = true;
}

void
Atom::backtrackFromBTrue ()
{
  Follows *f;
  for (f = pos; f != endPos; f++)
    f->r->backtrackFromTrueInPos (f->a);
  for (f = neg; f != endNeg; f++)
    f->r->backtrackFromTrueInNeg (f->a);
  Bpos = false;
  closure = true;
}

void
Atom::backtrackFromBFalse ()
{
  Follows *f;
  for (f = neg; f != endNeg; f++)
    f->r->backtrackFromFalseInNeg (f->a);
  for (f = pos; f != endPos; f++)
    f->r->backtrackFromFalseInPos (f->a);
  Bneg = false;
  closure = true;
}

void 
Atom::backchainFalse ()
{
  for (Follows *f = head; f != endHead; f++)
    f->r->backChainFalse ();
}

void
Atom::reduce_head ()
{
  Follows t;
  Follows *g = head;
  for (Follows *f = head; f != endHead; f++)
    {
      f->r->swapping (f, g);
      g->r->swapping (g, f);
      t = *g;
      *g = *f;
      *f = t;
      if (!g->r->isInactive ())
	g++;
    }
  endHead = g;
}

void
Atom::reduce_pbody (bool strong)
{
  Follows t;
  Follows *g = pos;
  for (Follows *f = pos; f != endPos; f++)
    {
      f->r->swapping (f, g);
      g->r->swapping (g, f);
      t = *g;
      *g = *f;
      *f = t;
      if ((strong == false || Bpos == false) &&
	  Bneg == false && !g->r->isInactive ())
	g++;
    }
  endPos = g;
  endPos2 = g;
}

void
Atom::reduce_nbody (bool strong)
{
  Follows t;
  Follows *g = neg;
  for (Follows *f = neg; f != endNeg; f++)
    {
      f->r->swapping (f, g);
      g->r->swapping (g, f);
      t = *g;
      *g = *f;
      *f = t;
      if (Bpos == false && Bneg == false && !g->r->isInactive ())
	g++;
    }
  endNeg = g;
}

void
Atom::reduce (bool strongly)
{
  reduce_head ();
  reduce_pbody (strongly);
  reduce_nbody (strongly);
}

void
Atom::unreduce ()
{
  endHead = pos;
  endPos = neg;
  endPos2 = neg;
  endNeg = end;
}

const char *
Atom::atom_name ()
{
  if (name)
    return name;
  else
    return "#noname#";
}

void
Atom::visit ()
{
  p->queue.push (this);
  visited = true;
}

BasicRule::BasicRule ()
{
  head = 0;
  neg = 0;
  pos = 0;
  pos2 = 0;
  nbody = 0;
  pbody = 0;
  nend = 0;
  pend = 0;
  end = 0;
  inactive = 0;
  visited = false;
  type = BASICRULE;
}

BasicRule::~BasicRule ()
{
  delete[] nbody;  // pbody is allocated after nbody
}

bool
BasicRule::isInactive ()
{
  return inactive;
}

bool
BasicRule::isUpperActive ()
{
  return pos2 == 0 && !inactive;
}

void
BasicRule::inactivate (Atom **a)
{
  inactive++;
  if (inactive == 1)
    {
      Atom *b = head;
      b->headof--;
      if (b->Bneg == false)
	{
	  if (b->headof && (b->source == 0 || b->source == this))
	    {
	      b->source = 0;
	      b->queue_push ();
	    }
	  if (b->headof == 0)
	    b->efalse_push ();
	  else if (b->headof == 1 && b->Bpos && *a != b)
	    b->backchainTrue ();
	}
    }
}

void
BasicRule::inactivateInTrue (Atom **a)
{
  if ((*a)->closure)
    pos2++;
  inactivate (a);
}

void
BasicRule::inactivateInFalse (Atom **a)
{
  inactivate (a);
}

void
BasicRule::fireInit ()
{
  if (pos == 0 && neg == 0)
    {
      head->etrue_push ();
      head->queue_push ();
      head->source = this;
    }
  else if (pos2 == 0)
    {
      head->queue_push ();
      if (head->source == 0)
	head->source = this;
    }
}

void
BasicRule::mightFireInitTrue (Atom **)
{
  pos2--;
  if (pos2 == 0 && !inactive)
    {
      head->queue_push ();
      if (head->source == 0)
	head->source = this;
    }
}

void
BasicRule::unInit ()
{
  pos2 = pos;
}

void
BasicRule::mightFire ()
{
  if (pos == 0)
    {
      if (neg == 0 && head->Bpos == false)
	head->etrue_push ();
      else if (neg == 1 && head->Bneg)
	backChainFalse ();
    }
  else if (pos == 1 && neg == 0 && head->Bneg)
    backChainFalse ();
}

void
BasicRule::mightFireTrue (Atom **)
{
  pos--;
  mightFire ();
}

void
BasicRule::mightFireFalse (Atom **)
{
  neg--;
  mightFire ();
}

void
BasicRule::backChainTrue ()
{
  Atom **b;
  if (neg)
    for (b = nbody; b != nend; b++)
      if ((*b)->Bneg == false)
	(*b)->efalse_push ();
  if (pos)
    for (b = pbody; b != pend; b++)
      if ((*b)->Bpos == false)
	(*b)->etrue_push ();
}

void
BasicRule::backChainFalse ()
{
  Atom **b;
  if (inactive)
    return;
  if (pos == 0 && neg == 1)
    {
      for (b = nbody; b != nend; b++)
	if ((*b)->Bneg == false)
	  {
	    (*b)->etrue_push ();
	    break;
	  }
    }
  else if (pos == 1 && neg == 0)
    for (b = pbody; b != pend; b++)
      if ((*b)->Bpos == false)
	{
	  (*b)->efalse_push ();
	  break;
	}
}

void
BasicRule::backtrackFromTrueInPos (Atom **)
{
  pos++;
}

void
BasicRule::backtrackFromFalseInPos (Atom **)
{
  pos2--;
  inactive--;
  if (inactive == 0)
    head->headof++;
}

void
BasicRule::backtrackFromTrueInNeg (Atom **)
{
  inactive--;
  if (inactive == 0)
    head->headof++;
}

void
BasicRule::backtrackFromFalseInNeg (Atom **)
{
  neg++;
}

void
BasicRule::propagateFalse (Atom **)
{
  pos2++;
  if (pos2 == 1 && !inactive && head->closure != false &&
      (head->source == 0 || head->source == this))
    {
      head->source = 0;
      head->queue_push ();
    }
}

void
BasicRule::propagateTrue (Atom **)
{
  pos2--;
  if (pos2 == 0 && !inactive && head->closure == false)
    {
      if (head->source == 0)
	head->source = this;
      head->queue_push ();
    }
}

void
BasicRule::backtrackUpper (Atom **)
{
  pos2--;
}

void
BasicRule::search (Atom *)
{
  if (!head->visited && head->Bneg == false)
    head->visit ();
  for (Atom **a = nbody; a != nend; a++)
    if (!(*a)->visited && (*a)->Bneg == false)
      (*a)->visit ();
  for (Atom **a = pbody; a != pend; a++)
    if (!(*a)->visited && (*a)->Bneg == false)
      (*a)->visit ();
  visited = true;
}

void
BasicRule::reduce (bool strongly)
{
  Atom **a;
  Atom **b;
  Atom *t;
  b = nbody;
  for (a = nbody; a != nend; a++)
    {
      t = *b;
      *b = *a;
      *a = t;
      if ((*b)->Bpos == false && (*b)->Bneg == false)
	b++;
    }
  nend = b;
  b = pbody;
  for (a = pbody; a != pend; a++)
    {
      t = *b;
      *b = *a;
      *a = t;
      if ((strongly == false || (*b)->Bpos == false)
	  && (*b)->Bneg == false)
	b++;
    }
  pend = b;
}

void
BasicRule::unreduce ()
{
  nend = pbody;
  pend = end;
}

void
BasicRule::setup ()
{
  head->head->r = this;
  head->head->a = &head;
  head->head++;
  Atom **a;
  for (a = nbody; a != nend; a++)
    {
      (*a)->negScore--;
      (*a)->neg[(*a)->negScore].r = this;
      (*a)->neg[(*a)->negScore].a = a;
    }
  for (a = pbody; a != pend; a++)
    {
      (*a)->posScore--;
      (*a)->pos[(*a)->posScore].r = this;
      (*a)->pos[(*a)->posScore].a = a;
    }
}

void
BasicRule::print ()
{
  cout << head->atom_name ();
  if (nbody)
    cout << " :- ";
  Atom **a;
  int comma = 0;
  for (a = pbody; a != pend; a++)
    {
      if (comma)
	cout << ", ";
      cout << (*a)->atom_name ();
      comma = 1;
    }
  for (a = nbody; a != nend; a++)
    {
      if (comma)
	cout << ", ";
      cout << "not " << (*a)->atom_name ();
      comma = 1;
    }
  cout << '.' << endl;
}


ConstraintRule::ConstraintRule ()
{
  head = 0;
  neg = 0;
  pos = 0;
  pos2 = 0;
  nbody = 0;
  pbody = 0;
  nend = 0;
  pend = 0;
  end = 0;
  inactivePos = 0;
  inactiveNeg = 0;
  visited = false;
  type = CONSTRAINTRULE;
}

ConstraintRule::~ConstraintRule ()
{
  delete[] nbody;
}

bool
ConstraintRule::isInactive ()
{
  return inactivePos > 0 || inactiveNeg > 0;
}

bool
ConstraintRule::isUpperActive ()
{
  return pos2 <= 0 && !isInactive ();
}

void
ConstraintRule::inactivate (Atom **a)
{
  Atom *b = head;
  b->headof--;
  if (b->Bneg == false)
    {
      if (b->headof && (b->source == 0 || b->source == this))
	{
	  b->source = 0;
	  b->queue_push ();
	}
      if (b->headof == 0)
	b->efalse_push ();
      else if (b->headof == 1 && b->Bpos && *a != b)
	b->backchainTrue ();
    }
}

void
ConstraintRule::inactivateInTrue (Atom **a)
{
  if ((*a)->closure)
    pos2++;
  inactivePos++;
  if (inactivePos == 1 && inactiveNeg <= 0)
    inactivate (a);
}

void
ConstraintRule::inactivateInFalse (Atom **a)
{
  inactiveNeg++;
  if (inactiveNeg == 1 && inactivePos <= 0)
    inactivate (a);
}

void
ConstraintRule::fireInit ()
{
  if (pos <= 0 && neg <= 0)
    {
      head->etrue_push ();
      head->queue_push ();
      head->source = this;
    }
  else if (pos2 <= 0)
    {
      head->queue_push ();
      if (head->source == 0)
	head->source = this;
    }
}

void
ConstraintRule::mightFireInitTrue (Atom **)
{
  pos2--;
  if (pos2 == 0 && !isInactive ())
    {
      head->queue_push ();
      if (head->source == 0)
	head->source = this;
    }
}

void
ConstraintRule::unInit ()
{
  pos2 = pos;
}

void
ConstraintRule::mightFireTrue (Atom **)
{
  pos--;
  if (pos == 0)
    {
      if (neg <= 0 && head->Bpos == false)
	head->etrue_push ();
      else if (neg == 1 && head->Bneg)
	backChainFalse ();
    }
  else if (pos == 1 && neg <= 0 && head->Bneg)
    backChainFalse ();
}

void
ConstraintRule::mightFireFalse (Atom **)
{
  neg--;
  if (neg <= 0)
    {
      if (pos <= 0 && head->Bpos == false)
	head->etrue_push ();
      else if (pos == 1 && head->Bneg)
	backChainFalse ();
    }
  else if (neg == 1 && pos <= 0 && head->Bneg)
    backChainFalse ();
}

void
ConstraintRule::backChainTrue ()
{
  if (inactivePos != 0 && inactiveNeg != 0)
    return;
  Atom **b;
  if (neg && inactiveNeg == 0)
    for (b = nbody; b != nend; b++)
      if ((*b)->Bneg == false && (*b)->Bpos == false)
	(*b)->efalse_push ();
  if (pos && inactivePos == 0)
    for (b = pbody; b != pend; b++)
      if ((*b)->Bneg == false && (*b)->Bpos == false)
	(*b)->etrue_push ();
}

void
ConstraintRule::backChainFalse ()
{
  Atom **b;
  if (isInactive ())
    return;
  if (pos <= 0 && neg == 1)
    {
      for (b = nbody; b != nend; b++)
	if ((*b)->Bneg == false && (*b)->Bpos == false)
	  (*b)->etrue_push ();
    }
  else if (pos == 1 && neg <= 0)
    for (b = pbody; b != pend; b++)
      if ((*b)->Bneg == false && (*b)->Bpos == false)
	(*b)->efalse_push ();
}

void
ConstraintRule::backtrackFromTrueInPos (Atom **)
{
  pos++;
}

void
ConstraintRule::backtrackFromFalseInPos (Atom **)
{
  pos2--;
  inactivePos--;
  if (inactivePos == 0 && inactiveNeg <= 0)
    head->headof++;
}

void
ConstraintRule::backtrackFromTrueInNeg (Atom **)
{
  inactiveNeg--;
  if (inactiveNeg == 0 && inactivePos <= 0)
    head->headof++;
}

void
ConstraintRule::backtrackFromFalseInNeg (Atom **)
{
  neg++;
}

void
ConstraintRule::propagateFalse (Atom **)
{
  pos2++;
  if (pos2 == 1 && !isInactive () && head->closure != false &&
      (head->source == 0 || head->source == this))
    {
      head->source = 0;
      head->queue_push ();
    }
}

void
ConstraintRule::propagateTrue (Atom **)
{
  pos2--;
  if (pos2 == 0 && !isInactive () && head->closure == false)
    {
      if (head->source == 0)
	head->source = this;
      head->queue_push ();
    }
}

void
ConstraintRule::backtrackUpper (Atom **)
{
  pos2--;
}

void
ConstraintRule::search (Atom *)
{
  if (!head->visited && head->Bneg == false)
    head->visit ();
  for (Atom **a = nbody; a != nend; a++)
    if (!(*a)->visited && (*a)->Bneg == false)
      (*a)->visit ();
  for (Atom **a = pbody; a != pend; a++)
    if (!(*a)->visited && (*a)->Bneg == false)
      (*a)->visit ();
  visited = true;
}

void
ConstraintRule::reduce (bool strongly)
{
  Atom **a;
  Atom **b;
  Atom *t;
  b = nbody;
  for (a = nbody; a != nend; a++)
    {
      t = *b;
      *b = *a;
      *a = t;
      if ((*b)->Bpos == false && (*b)->Bneg == false)
	b++;
    }
  nend = b;
  b = pbody;
  for (a = pbody; a != pend; a++)
    {
      t = *b;
      *b = *a;
      *a = t;
      if ((strongly == false || (*b)->Bpos == false) &&
	  (*b)->Bneg == false)
	b++;
    }
  pend = b;
}

void
ConstraintRule::unreduce ()
{
  nend = pbody;
  pend = end;
}

void
ConstraintRule::setup ()
{
  head->head->r = this;
  head->head->a = &head;
  head->head++;
  Atom **a;
  for (a = nbody; a != nend; a++)
    {
      (*a)->negScore--;
      (*a)->neg[(*a)->negScore].r = this;
      (*a)->neg[(*a)->negScore].a = a;
    }
  for (a = pbody; a != pend; a++)
    {
      (*a)->posScore--;
      (*a)->pos[(*a)->posScore].r = this;
      (*a)->pos[(*a)->posScore].a = a;
    }
}

void
ConstraintRule::print ()
{
  cout << head->atom_name () << " :- ";
  Atom **a;
  int comma = 0;
  if (pbody != pend)
    {
      long atleast = pos;
      for (a = pbody; a != pend; a++)
	if ((*a)->Bpos)
	  atleast++;
      cout << atleast << " { ";
      for (a = pbody; a != pend; a++)
	{
	  if (comma)
	    cout << ", ";
	  cout << (*a)->atom_name ();
	  comma = 1;
	}
      cout << " }";
    }
  if (nbody != nend)
    {
      if (comma)
	cout << ' ';
      comma = 0;
      long atleast = neg;
      for (a = nbody; a != nend; a++)
	if ((*a)->Bneg)
	  atleast++;
      cout << atleast << " { ";
      for (a = nbody; a != nend; a++)
	{
	  if (comma)
	    cout << ", ";
	  cout << "not " << (*a)->atom_name ();
	  comma = 1;
	}
      cout << " }";
    }
  cout << '.' << endl;
}

GenerateRule::GenerateRule ()
{
  head = 0;
  hend = 0;
  pbody = 0;
  pend = 0;
  end = 0;
  neg = 0;
  pos = 0;
  pos2 = 0;
  inactivePos = 0;
  inactiveNeg = 0;
  visited = false;
  type = GENERATERULE;
}

GenerateRule::~GenerateRule ()
{
  delete[] head;
}

bool
GenerateRule::isInactive ()
{
  return inactivePos || inactiveNeg > 0;
}

bool
GenerateRule::isUpperActive ()
{
  return pos2 == 0 && !isInactive ();
}

void
GenerateRule::inactivate (Atom **a)
{
  for (Atom **h = head; h != hend; h++)
    {
      Atom *b = *h;
      b->headof--;
      if (b->Bneg == false)
	{
	  if (b->headof && (b->source == 0 || b->source == this))
	    {
	      b->source = 0;
	      b->queue_push ();
	    }
	  if (b->headof == 0)
	    b->efalse_push ();
	  else if (b->headof == 1 && b->Bpos && *a != b)
	    b->backchainTrue ();
	}
    }
}

void
GenerateRule::inactivateInTrue (Atom **a)
{
  if ((*a)->closure)
    pos2++;
  inactivePos++;
  if (inactivePos == 1 && inactiveNeg <= 0)
    inactivate (a);
}

void
GenerateRule::inactivateInFalse (Atom **a)
{
  inactiveNeg++;
  if (inactiveNeg == 1 && inactivePos == 0)
    inactivate (a);
}

void
GenerateRule::fireInit ()
{
  if (pos == 0 && neg <= 0)
    for (Atom **h = head; h != hend; h++)
      {
	(*h)->etrue_push ();
	(*h)->queue_push ();
	(*h)->source = this;
      }
  else if (pos2 == 0)
    for (Atom **h = head; h != hend; h++)
      {
	(*h)->queue_push ();
	if ((*h)->source == 0)
	  (*h)->source = this;
      }
}

void
GenerateRule::mightFireInitTrue (Atom **)
{
  pos2--;
  if (pos2 == 0 && !isInactive ())
    for (Atom **h = head; h != hend; h++)
      {
	(*h)->queue_push ();
	if ((*h)->source == 0)
	  (*h)->source = this;
      }
}

void
GenerateRule::unInit ()
{
  pos2 = pos;
}

void
GenerateRule::mightFire ()
{
  if (pos == 0 && neg == 0)
    {
      for (Atom **h = head; h != hend; h++)
	if ((*h)->Bneg == false && (*h)->Bpos == false)
	  (*h)->etrue_push ();
    }
  else if (pos == 1 && neg < 0)
    backChainFalse ();
}

void
GenerateRule::mightFireTrue (Atom **)
{
  pos--;
  mightFire ();
}

void
GenerateRule::mightFireFalse (Atom **)
{
  neg--;
  mightFire ();
}

void
GenerateRule::backChainTrue ()
{
  if (inactiveNeg != 0)
    return;
  Atom **b;
  if (neg)
    for (b = head; b != hend; b++)
      if ((*b)->Bneg == false && (*b)->Bpos == false)
	(*b)->efalse_push ();
  if (pos)
    for (b = pbody; b != pend; b++)
      if ((*b)->Bpos == false)
	(*b)->etrue_push ();
}

void
GenerateRule::backChainFalse ()
{
  if (isInactive ())
    return;
  if (pos == 1 && neg < 0)
    for (Atom **b = pbody; b != pend; b++)
      if ((*b)->Bpos == false)
	{
	  (*b)->efalse_push ();
	  break;
	}
}

void
GenerateRule::backtrackFromTrueInPos (Atom **)
{
  pos++;
}

void
GenerateRule::backtrackFromFalseInPos (Atom **)
{
  pos2--;
  inactivePos--;
  if (inactivePos == 0 && inactiveNeg <= 0)
    for (Atom **h = head; h != hend; h++)
      (*h)->headof++;
}

void
GenerateRule::backtrackFromTrueInNeg (Atom **)
{
  inactiveNeg--;
  if (inactiveNeg == 0 && inactivePos == 0)
    for (Atom **h = head; h != hend; h++)
      (*h)->headof++;
}

void
GenerateRule::backtrackFromFalseInNeg (Atom **)
{
  neg++;
}

void
GenerateRule::propagateFalse (Atom **)
{
  pos2++;
  if (pos2 == 1 && !isInactive ())
    {
      if (inactiveNeg < 0)
	{
	  for (Atom **h = head; h != hend; h++)
	    if ((*h)->closure != false &&
		((*h)->source == 0 || (*h)->source == this))
	      {
		(*h)->source = 0;
		(*h)->queue_push ();
	      }
	}
      else
	for (Atom **h = head; h != hend; h++)
	  if ((*h)->closure != false && (*h)->Bpos &&
	      ((*h)->source == 0 || (*h)->source == this))
	    {
	      (*h)->source = 0;
	      (*h)->queue_push ();
	    }
    }
}

void
GenerateRule::propagateTrue (Atom **)
{
  pos2--;
  if (pos2 == 0 && !isInactive ())
    {
      if (inactiveNeg < 0)
	{
	  for (Atom **h = head; h != hend; h++)
	    if ((*h)->closure == false)
	      {
		if ((*h)->source == 0)
		  (*h)->source = this;
		(*h)->queue_push ();
	      }
	}
      else
	for (Atom **h = head; h != hend; h++)
	  if ((*h)->closure == false && (*h)->Bpos)
	    {
	      if ((*h)->source == 0)
		(*h)->source = this;
	      (*h)->queue_push ();
	    }
    }
}

void
GenerateRule::backtrackUpper (Atom **)
{
  pos2--;
}

void
GenerateRule::search (Atom *)
{
  for (Atom **a = head; a != hend; a++)
    if (!(*a)->visited && (*a)->Bneg == false)
      (*a)->visit ();
  for (Atom **a = pbody; a != pend; a++)
    if (!(*a)->visited && (*a)->Bneg == false)
      (*a)->visit ();
  visited = true;
}

void
GenerateRule::reduce (bool strongly)
{
  Atom **a;
  Atom **b;
  Atom *t;
  b = head;
  for (a = head; a != hend; a++)
    {
      t = *b;
      *b = *a;
      *a = t;
      if ((*b)->Bpos == false && (*b)->Bneg == false)
	b++;
    }
  hend = b;
  b = pbody;
  for (a = pbody; a != pend; a++)
    {
      t = *b;
      *b = *a;
      *a = t;
      if ((strongly == false || (*b)->Bpos == false) &&
	  (*b)->Bneg == false)
	b++;
    }
  pend = b;
}

void
GenerateRule::unreduce ()
{
  hend = pbody;
  pend = end;
}

void
GenerateRule::setup ()
{
  Atom **a;
  for (a = head; a != hend; a++)
    {
      (*a)->head->r = this;
      (*a)->head->a = a;
      (*a)->head++;
    }
  for (a = head; a != hend; a++)
    {
      (*a)->negScore--;
      (*a)->neg[(*a)->negScore].r = this;
      (*a)->neg[(*a)->negScore].a = a;
    }
  for (a = pbody; a != pend; a++)
    {
      (*a)->posScore--;
      (*a)->pos[(*a)->posScore].r = this;
      (*a)->pos[(*a)->posScore].a = a;
    }
}

void
GenerateRule::print ()
{
  Atom **a;
  int comma = 0;
  long atleast = inactiveNeg;
  for (a = head; a != hend; a++)
    if ((*a)->Bpos == true)
      atleast--;
  cout << -atleast << " {";
  for (a = head; a != hend; a++)
    {
      if (comma)
	cout << ", ";
      cout << (*a)->atom_name ();
      comma = 1;
    }
  cout << '}';
  if (pbody != pend)
    cout << " :- ";
  comma = 0;
  for (a = pbody; a != pend; a++)
    {
      if (comma)
	cout << ", ";
      cout << (*a)->atom_name ();
      comma = 1;
    }
  cout << '.' << endl;
}

ChoiceRule::ChoiceRule ()
{
  head = 0;
  hend = 0;
  neg = 0;
  pos = 0;
  pos2 = 0;
  nbody = 0;
  pbody = 0;
  nend = 0;
  pend = 0;
  end = 0;
  inactive = 0;
  visited = false;
  type = CHOICERULE;
}

ChoiceRule::~ChoiceRule ()
{
  delete[] head;
}

bool
ChoiceRule::isInactive ()
{
  return inactive;
}

bool
ChoiceRule::isUpperActive ()
{
  return pos2 == 0 && !inactive;
}

void
ChoiceRule::inactivate (Atom **a)
{
  inactive++;
  if (inactive == 1)
    {
      for (Atom **h = head; h != hend; h++)
	{
	  Atom *b = *h;
	  b->headof--;
	  if (b->Bneg == false)
	    {
	      if (b->headof && (b->source == 0 || b->source == this))
		{
		  b->source = 0;
		  b->queue_push ();
		}
	      if (b->headof == 0)
		b->efalse_push ();
	      else if (b->headof == 1 && b->Bpos && *a != b)
		b->backchainTrue ();
	    }
	}
    }
}

void
ChoiceRule::inactivateInTrue (Atom **a)
{
  if ((*a)->closure)
    pos2++;
  inactivate (a);
}

void
ChoiceRule::inactivateInFalse (Atom **a)
{
  inactivate (a);
}

void
ChoiceRule::fireInit ()
{
  if ((pos == 0 && neg == 0) || pos2 == 0)
    for (Atom **h = head; h != hend; h++)
      {
	(*h)->queue_push ();
	if ((*h)->source == 0)
	  (*h)->source = this;
      }
}

void
ChoiceRule::mightFireInitTrue (Atom **)
{
  pos2--;
  if (pos2 == 0 && !inactive)
    for (Atom **h = head; h != hend; h++)
      {
	(*h)->queue_push ();
	if ((*h)->source == 0)
	  (*h)->source = this;
      }
}

void
ChoiceRule::unInit ()
{
  pos2 = pos;
}

void
ChoiceRule::mightFire ()
{
}

void
ChoiceRule::mightFireTrue (Atom **)
{
  pos--;
}

void
ChoiceRule::mightFireFalse (Atom **)
{
  neg--;
}

void
ChoiceRule::backChainTrue ()
{
  Atom **b;
  if (neg)
    for (b = nbody; b != nend; b++)
      if ((*b)->Bneg == false)
	(*b)->efalse_push ();
  if (pos)
    for (b = pbody; b != pend; b++)
      if ((*b)->Bpos == false)
	(*b)->etrue_push ();
}

void
ChoiceRule::backChainFalse ()
{
}

void
ChoiceRule::backtrackFromTrueInPos (Atom **)
{
  pos++;
}

void
ChoiceRule::backtrackFromFalseInPos (Atom **)
{
  pos2--;
  inactive--;
  if (inactive == 0)
    for (Atom **h = head; h != hend; h++)
      (*h)->headof++;
}

void
ChoiceRule::backtrackFromTrueInNeg (Atom **)
{
  inactive--;
  if (inactive == 0)
    for (Atom **h = head; h != hend; h++)
      (*h)->headof++;
}

void
ChoiceRule::backtrackFromFalseInNeg (Atom **)
{
  neg++;
}

void
ChoiceRule::propagateFalse (Atom **)
{
  pos2++;
  if (pos2 == 1 && !inactive)
    for (Atom **h = head; h != hend; h++)
      if ((*h)->closure != false &&
	  ((*h)->source == 0 || (*h)->source == this))
	{
	  (*h)->source = 0;
	  (*h)->queue_push ();
	}
}

void
ChoiceRule::propagateTrue (Atom **)
{
  pos2--;
  if (pos2 == 0 && !inactive)
    for (Atom **h = head; h != hend; h++)
      if ((*h)->closure == false)
	{
	  if ((*h)->source == 0)
	    (*h)->source = this;
	  (*h)->queue_push ();
	}
}

void
ChoiceRule::backtrackUpper (Atom **)
{
  pos2--;
}

void
ChoiceRule::search (Atom *)
{
  for (Atom **h = head; h != hend; h++)
    if (!(*h)->visited && (*h)->Bneg == false)
      (*h)->visit ();
  for (Atom **a = nbody; a != nend; a++)
    if (!(*a)->visited && (*a)->Bneg == false)
      (*a)->visit ();
  for (Atom **a = pbody; a != pend; a++)
    if (!(*a)->visited && (*a)->Bneg == false)
      (*a)->visit ();
  visited = true;
}

void
ChoiceRule::reduce (bool strongly)
{
  Atom **a;
  Atom **b;
  Atom *t;
  b = head;
  for (a = head; a != hend; a++)
    {
      t = *b;
      *b = *a;
      *a = t;
      if ((strongly == false || (*b)->Bpos == false)
	  && (*b)->Bneg == false)
	b++;
    }
  hend = b;
  b = nbody;
  for (a = nbody; a != nend; a++)
    {
      t = *b;
      *b = *a;
      *a = t;
      if ((*b)->Bpos == false && (*b)->Bneg == false)
	b++;
    }
  nend = b;
  b = pbody;
  for (a = pbody; a != pend; a++)
    {
      t = *b;
      *b = *a;
      *a = t;
      if ((strongly == false || (*b)->Bpos == false)
	  && (*b)->Bneg == false)
	b++;
    }
  pend = b;
}

void
ChoiceRule::unreduce ()
{
  hend = nbody;
  nend = pbody;
  pend = end;
}

void
ChoiceRule::setup ()
{
  Atom **a;
  for (a = head; a != hend; a++)
    {
      (*a)->head->r = this;
      (*a)->head->a = a;
      (*a)->head++;
    }
  for (a = nbody; a != nend; a++)
    {
      (*a)->negScore--;
      (*a)->neg[(*a)->negScore].r = this;
      (*a)->neg[(*a)->negScore].a = a;
    }
  for (a = pbody; a != pend; a++)
    {
      (*a)->posScore--;
      (*a)->pos[(*a)->posScore].r = this;
      (*a)->pos[(*a)->posScore].a = a;
    }
}

void
ChoiceRule::print ()
{
  Atom **a;
  bool comma = false;
  cout << "{ ";
  for (a = head; a != hend; a++)
    {
      if (comma)
	cout << ", ";
      cout << (*a)->atom_name ();
      comma = true;
    }
  cout << " }";
  comma = false;
  if (pbody != pend || nbody != nend)
    cout << " :- ";
  for (a = pbody; a != pend; a++)
    {
      if (comma)
	cout << ", ";
      cout << (*a)->atom_name ();
      comma = true;
    }
  for (a = nbody; a != nend; a++)
    {
      if (comma)
	cout << ", ";
      cout << "not " << (*a)->atom_name ();
      comma = true;
    }
  cout << '.' << endl;
}


WeightRule::WeightRule ()
{
  head = 0;
  bend = 0;
  body = 0;
  end = 0;
  reverse = 0;
  weight = 0;
  positive = 0;
  minweight = 0;
  maxweight = 0;
  upper_min = 0;
  atleast = 0;
  lower = 0;
  upper = 0;
  lower_shadow = 0;
  upper_shadow = 0;
  visited = false;
  type = WEIGHTRULE;
}

WeightRule::~WeightRule ()
{
  delete[] body;
  delete[] weight;
  delete[] positive;
  delete[] reverse;
}

bool
WeightRule::isInactive ()
{
  return maxweight < atleast;
}

bool
WeightRule::isUpperActive ()
{
  return upper_min >= atleast;
}

bool
WeightRule::fired ()
{
  return minweight >= atleast;
}

void
WeightRule::change (Atom *a)
{
  if (isInactive ())
    {
      Atom *b = head;
      b->headof--;
      if (b->Bneg == false)
	{
	  if (b->headof && (b->source == 0 || b->source == this))
	    {
	      b->source = 0;
	      b->queue_push ();
	    }
	  if (b->headof == 0)
	    b->efalse_push ();
	  else if (b->headof == 1 && b->Bpos && a != b)
	    b->backchainTrue ();
	}
    }
  else if (fired ())
    {
      if (head->Bpos == false)
	head->etrue_push ();
    }
  else if (head->Bneg)
    backChainFalse ();
  else if (head->Bpos && head->headof == 1)
    backChainTrue ();
}

void
WeightRule::upper_lower (Atom **a)
{
  if (upper == a)
    do 
      upper++;
    while (upper != bend && ((*upper)->Bpos || (*upper)->Bneg));
  if (lower == a)
    do
      lower++;
    while (lower != bend && ((*lower)->Bpos || (*lower)->Bneg));
}

void
WeightRule::inactivate (Atom **a)
{
  bool inactive = isInactive ();
  bool frd = fired ();
  maxweight -= weight[a-body];
  upper_lower (a);
  if (!frd && !inactive)
    change (*a);
}

void
WeightRule::inactivateInTrue (Atom **a)
{
  if ((*a)->closure)
    upper_min -= weight[a-body];
  inactivate (a);
}

void
WeightRule::inactivateInFalse (Atom **a)
{
  inactivate (a);
}

void
WeightRule::fireInit ()
{
  if (fired ())
    {
      head->etrue_push ();
      head->queue_push ();
      head->source = this;
    }
  else if (isUpperActive () && !isInactive ())
    {
      head->queue_push ();
      if (head->source == 0)
	head->source = this;
    }
}

void
WeightRule::mightFireInitTrue (Atom **a)
{
  bool active = isUpperActive ();
  upper_min += weight[a-body];
  if (!active && isUpperActive () && !isInactive ())
    {
      head->queue_push ();
      if (head->source == 0)
	head->source = this;
    }
}

void
WeightRule::unInit ()
{
  upper_min = 0;
  for (Atom **a = body; a != bend; a++)
    if (!positive[a-body])
      upper_min += weight[a-body];
}

void
WeightRule::mightFire (Atom **a)
{
  bool inactive = isInactive ();
  bool frd = fired ();
  minweight += weight[a-body];
  upper_lower (a);
  if (!frd && !inactive)
    change (*a);
}

void
WeightRule::mightFireTrue (Atom **a)
{
  mightFire (a);
}

void
WeightRule::mightFireFalse (Atom **a)
{
  mightFire (a);
}

void
WeightRule::backChainTrue ()
{
  if (fired ())
     return;
  for (; upper_shadow != bend; upper_shadow++)
    {
      if ((*upper_shadow)->Bpos || (*upper_shadow)->Bneg)
	continue;
      if (maxweight - weight[upper_shadow-body] < atleast)
	{
	  if (positive[upper_shadow-body])
	    (*upper_shadow)->etrue_push ();
	  else
	    (*upper_shadow)->efalse_push ();
	}
      else
	break;
    }
}

void
WeightRule::backChainFalse ()
{
  if (isInactive () || fired ())
    return;
  for (; lower_shadow != bend; lower_shadow++)
    {
      if ((*lower_shadow)->Bpos || (*lower_shadow)->Bneg)
	continue;
      if (minweight + weight[lower_shadow-body] >= atleast)
	{
	  if (positive[lower_shadow-body])
	    (*lower_shadow)->efalse_push ();
	  else
	    (*lower_shadow)->etrue_push ();
	}
      else
	break;
    }
}

void
WeightRule::backtrack (Atom **a)
{
  if (upper > a)
    upper = a;
  upper_shadow = upper;
  if (lower > a)
    lower = a;
  lower_shadow = lower;
}

void
WeightRule::backtrackFromTrueInPos (Atom **a)
{
  minweight -= weight[a-body];
  backtrack (a);
}

void
WeightRule::backtrackFromFalseInPos (Atom **a)
{
  bool inactive = isInactive ();
  Weight w = weight[a-body];
  maxweight += w;
  upper_min += w;
  if (inactive && !isInactive ())
    head->headof++;
  backtrack (a);
}

void
WeightRule::backtrackFromTrueInNeg (Atom **a)
{
  bool inactive = isInactive ();
  maxweight += weight[a-body];
  if (inactive && !isInactive ())
    head->headof++;
  backtrack (a);
}

void
WeightRule::backtrackFromFalseInNeg (Atom **a)
{
  minweight -= weight[a-body];
  backtrack (a);
}

void
WeightRule::propagateFalse (Atom **a)
{
  bool active = isUpperActive ();
  bool inactive = isInactive ();
  upper_min -= weight[a-body];
  if (!inactive && active && !isUpperActive () &&
      head->closure != false &&
      (head->source == 0 || head->source == this))
    {
      head->source = 0;
      head->queue_push ();
    }
}

void
WeightRule::propagateTrue (Atom **a)
{
  bool active = isUpperActive ();
  bool inactive = isInactive ();
  upper_min += weight[a-body];
  if (!inactive && !active && isUpperActive () &&
      head->closure == false)
    {
      if (head->source == 0)
	head->source = this;
      head->queue_push ();
    }
}

void
WeightRule::backtrackUpper (Atom **a)
{
  upper_min += weight[a-body];
}

void
WeightRule::search (Atom *a)
{
  if (a->Bneg || (a->Bpos && a != head))
    return;
  if (!head->visited && head->Bneg == false)
    head->visit ();
  for (Atom **b = body; b != bend; b++)
    if (!(*b)->visited && (*b)->Bneg == false)
      (*b)->visit ();
  visited = true;
}

void
WeightRule::reduce (bool strongly)
{
  Atom **b = body;
  for (Atom **a = body; a != bend; a++)
    {
      swap (b-body, a-body);
      if (positive[b-body])
	{
	  if ((strongly == false || (*b)->Bpos == false) &&
	      (*b)->Bneg == false)
	    b++;
	}
      else if ((*b)->Bpos == false && (*b)->Bneg == false)
	b++;
    }
  bend = b;
  backtrack (body);
}

void
WeightRule::unreduce ()
{
  bend = end;
  backtrack (body);
  sort ();
}

void
WeightRule::initWeight (Weight w)
{
  maxweight += w;
}

void
WeightRule::swap (long a, long b)
{
  Weight tw = weight[a];
  weight[a] = weight[b];
  weight[b] = tw;
  Atom *ta = body[a];
  body[a] = body[b];
  body[b] = ta;
  bool tb = positive[a];
  positive[a] = positive[b];
  positive[b] = tb;
  if (reverse[a])
    reverse[a]->a = body+b;
  if (reverse[b])
  reverse[b]->a = body+a;
}

bool
WeightRule::smaller (long a, long b)
{
  if (weight[a] < weight[b])
    return true;
  else
    return false;
}

void
WeightRule::heap (long k, long e)
{
  long a = 2*k+1;
  while (a < e)
    {
      if (smaller (a+1, a))
	a++;
      if (smaller (a, k))
	swap (a, k);
      else
	break;
      k = a;
      a = 2*k+1;
    }
  if (a == e && smaller (a, k))
    swap (a, k);
}

void
WeightRule::sort ()
{
  long i;
  long e = bend-body-1;
  for (i = (e-1)/2; i >= 0; i--)
    heap (i, e);
  i = e;
  while (i > 0)
    {
      swap (i, 0);
      i--;
      heap (0, i);
    }
}

void
WeightRule::swapping (Follows *f, Follows *g)
{
  if (f->a != &head)
    reverse[f->a - body] = g;
}

void
WeightRule::setup ()
{
  head->head->r = this;
  head->head->a = &head;
  head->head++;
  Atom **a;
  Weight *w = weight;
  sort ();
  for (a = body; a != bend; a++)
    if (positive[a-body])
      {
	(*a)->posScore--;
	(*a)->pos[(*a)->posScore].r = this;
	(*a)->pos[(*a)->posScore].a = a;
	reverse[a-body] = (*a)->pos + (*a)->posScore;
	initWeight (*w++);
      }
    else
      {
	(*a)->negScore--;
	(*a)->neg[(*a)->negScore].r = this;
	(*a)->neg[(*a)->negScore].a = a;
	reverse[a-body] = (*a)->neg + (*a)->negScore;
	upper_min += *w;
	initWeight (*w++);
      }
}

void
WeightRule::print ()
{
  cout << head->atom_name () << " :- { ";
  int comma = 0;
  for (Atom **a = body; a != bend; a++)
    if (positive[a-body])
      {
	if (comma)
	  cout << ", ";
	cout << (*a)->atom_name () << " = " << weight[a-body];
	comma = 1;
      }
    else
      {
	if (comma)
	  cout << ", ";
	cout << "not " << (*a)->atom_name () << " = "
	     << weight[a-body];
	comma = 1;
      }
  cout << "} >=" << atleast << '.' << endl;
}


OptimizeRule::OptimizeRule ()
{
  nbody = 0;
  pbody = 0;
  nend = 0;
  pend = 0;
  end = 0;
  weight = 0;
  minweight = 0;
  maxweight = 0;
  maxoptimum = 0;
  minoptimum = 0;
  visited = false;
  type = OPTIMIZERULE;
  maximize = false;
  next = 0;
}

OptimizeRule::~OptimizeRule ()
{
  delete[] nbody;
  delete[] weight;
}

void
OptimizeRule::setOptimum ()
{
  minoptimum = minweight;
  maxoptimum = maxweight;
}

bool
OptimizeRule::isInactive ()
{
  return false;  // Reduce(Strongly|Weakly) needs this
}

bool
OptimizeRule::isUpperActive ()
{
  return false;
}

void
OptimizeRule::inactivate (Atom **a)
{
  Weight w = weight[a-nbody];
  if (w >= 0)
    maxweight -= w;
  else
    minweight -= w;
}

void
OptimizeRule::inactivateInTrue (Atom **a)
{
  inactivate (a);
}

void
OptimizeRule::inactivateInFalse (Atom **a)
{
  inactivate (a);
}

void
OptimizeRule::fireInit ()
{
}

void
OptimizeRule::mightFireInitTrue (Atom **)
{
}

void
OptimizeRule::unInit ()
{
}

void
OptimizeRule::mightFire (Atom **a)
{
  Weight w = weight[a-nbody];
  if (w < 0)
    maxweight += w;
  else
    minweight += w;
}

void
OptimizeRule::mightFireTrue (Atom **a)
{
  mightFire (a);
}

void
OptimizeRule::mightFireFalse (Atom **a)
{
  mightFire (a);
}

void
OptimizeRule::backChainTrue ()
{
}

void
OptimizeRule::backChainFalse ()
{
}

void
OptimizeRule::backtrackFromTrueInPos (Atom **a)
{
  Weight w = weight[a-nbody];
  if (w < 0)
    maxweight -= w;
  else
    minweight -= w;
}

void
OptimizeRule::backtrackFromFalseInPos (Atom **a)
{
  Weight w = weight[a-nbody];
  if (w >= 0)
    maxweight += w;
  else
    minweight += w;
}

void
OptimizeRule::backtrackFromTrueInNeg (Atom **a)
{
  backtrackFromFalseInPos (a);
}

void
OptimizeRule::backtrackFromFalseInNeg (Atom **a)
{
  backtrackFromTrueInPos (a);
}

void
OptimizeRule::propagateFalse (Atom **)
{
}

void
OptimizeRule::propagateTrue (Atom **)
{
}

void
OptimizeRule::backtrackUpper (Atom **)
{
}

void
OptimizeRule::search (Atom *a)
{
  if (a->Bneg || a->Bpos)
    return;
  Atom **b;
  for (b = nbody; b != nend; b++)
    if (!(*b)->visited && (*b)->Bneg == false && (*b)->Bpos == false)
      (*b)->visit ();
  for (b = pbody; b != pend; b++)
    if (!(*b)->visited && (*b)->Bneg == false && (*b)->Bpos == false)
      (*b)->visit ();
  visited = true;
}

void
OptimizeRule::reduce (bool)
{
  Atom **a;
  Atom **b;
  Atom *t;
  b = nbody;
  for (a = nbody; a != nend; a++)
    {
      t = *b;
      *b = *a;
      *a = t;
      if ((*b)->Bpos == false && (*b)->Bneg == false)
	b++;
    }
  nend = b;
  b = pbody;
  for (a = pbody; a != pend; a++)
    {
      t = *b;
      *b = *a;
      *a = t;
      if ((*b)->Bpos == false && (*b)->Bneg == false)
	b++;
    }
  pend = b;
}

void
OptimizeRule::unreduce ()
{
  nend = pbody;
  pend = end;
}

void
OptimizeRule::initWeight (Weight w)
{
  if (w > 0)
    maxweight += w;
  else
    minweight += w;
}

void
OptimizeRule::setup ()
{
  Atom **a;
  Weight *w = weight;
  for (a = nbody; a != nend; a++)
    {
      (*a)->negScore--;
      (*a)->neg[(*a)->negScore].r = this;
      (*a)->neg[(*a)->negScore].a = a;
      initWeight (*w++);
    }
  for (a = pbody; a != pend; a++)
    {
      (*a)->posScore--;
      (*a)->pos[(*a)->posScore].r = this;
      (*a)->pos[(*a)->posScore].a = a;
      initWeight (*w++);
    }
}

void
OptimizeRule::print ()
{
  if (maximize)
    cout << "maximize { ";
  else
    cout << "minimize { ";
  Atom **a;
  int comma = 0;
  for (a = pbody; a != pend; a++)
    {
      if (comma)
	cout << ", ";
      cout << (*a)->atom_name () << " = " << weight[a-nbody];
      comma = 1;
    }
  for (a = nbody; a != nend; a++)
    {
      if (comma)
	cout << ", ";
      cout << "not " << (*a)->atom_name () << " = " << weight[a-nbody];
      comma = 1;
    }
  cout << " }" << endl;
}
