// Copyright 1998 by Patrik Simons
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston,
// MA 02111-1307, USA.
//
// Patrik.Simons@hut.fi
#ifndef ATOMRULE_H
#define ATOMRULE_H

#include "defines.h"

class Rule;
class Atom;
class Program;

struct Follows
{
  Rule *r;
  Atom **a;
};

class Obj
{
public:
  Obj () {};
  virtual ~Obj () {};
};

class Atom : public Obj
{
public:
  Atom (Program *p);
  ~Atom ();

  void setBTrue ();
  void setBFalse ();
  void setTrue ();
  void backtrackFromBTrue ();
  void backtrackFromBFalse ();
  void backchainTrue ();
  void backchainFalse ();
  void reduce_head ();
  void reduce_pbody (bool strong);
  void reduce_nbody (bool strong);
  void reduce (bool strongly);
  void unreduce ();
  const char *atom_name ();
  void etrue_push ();
  void efalse_push ();
  void queue_push ();
  void visit ();

  Program * const p;  // The program in which the atom appears
  Follows *endHead;   // Sentinel of the head array
  Follows *endPos;    // Sentinel of the pos array
  Follows *endNeg;    // Sentinel of the neg array 
  Follows *endNeg2;   // As below but for the neg array
  Follows *endPos2;   // Sentinel of the pos2 array = pos array, used
		      // to localize and thereby speed up the upper
		      // closure computation
  Follows *end;       // The end of the array
  long headof;        // The number of rules in head array whose
		      // inactive counter is zero
  Follows *head;        // The rules whose heads are atom
  Follows *pos;         // The rules in which atom appears positively
			// (allocated in head)
  Follows *neg;         // The rules in which atom appears negatively
			// (allocated in head)
  Rule *source;       // The rule that keeps this atom in upper closure
  long posScore;      // Used in heuristic and improvement of upper
		      // closure
  long negScore;      // Used in heuristic and improvement of upper
		      // closure
  char *name;         // The name of the atom
  bool closure : 1;       // True if the atom is in the upper closure
  bool Bpos : 1;          // True if the atom is in B+
  bool Bneg : 1;          // True if the atom is in B-
  bool visited : 1;       // For use in backjumping and improvement of
		          // upper closure
  bool guess : 1;         // True if the atom currently is a guess
  bool isnant : 1;        // True if the atom is in NAnt(P)
  bool dependsonTrue : 1; // Used by lookahead
  bool dependsonFalse : 1;// Used by lookahead
  bool computeTrue : 1;   // Compute statement
  bool computeFalse : 1;  // Compute statement
  bool backtracked : 1;   // For use in printStack
  bool forced : 1;        // For use in printStack
  bool in_etrue_queue : 1; // Atom is in equeue, will get Bpos == true
  bool in_efalse_queue : 1;// Atom is in equeue, will get Bneg == true
};

class Rule : public Obj
{
public:
  Rule () { visited = false; type = ENDRULE; };
  virtual ~Rule () {};
  
  virtual void inactivateInTrue (Atom **) = 0;
  virtual void inactivateInFalse (Atom **) = 0;
  virtual void fireInit () = 0;
  virtual void mightFireInitTrue (Atom **) = 0;
  virtual void mightFireInitFalse (Atom **) {};
  virtual void unInit () = 0;
  virtual void mightFireTrue (Atom **) = 0;
  virtual void mightFireFalse (Atom **) = 0;
  virtual void backChainTrue () = 0;
  virtual void backChainFalse () = 0;
  virtual void backtrackFromTrueInPos (Atom **) = 0;
  virtual void backtrackFromFalseInPos (Atom **) = 0;
  virtual void backtrackFromTrueInNeg (Atom **) = 0;
  virtual void backtrackFromFalseInNeg (Atom **) = 0;
  virtual void propagateFalse (Atom **) = 0;
  virtual void propagateTrue (Atom **) = 0;
  virtual void backtrackUpper (Atom **) = 0;
  virtual void propagateFalseInNeg (Atom **) {};
  virtual void propagateTrueInNeg (Atom **) {};
  virtual void backtrackUpperInNeg (Atom **) {};

  virtual bool isInactive () = 0;
  virtual bool isUpperActive () = 0;
  virtual void search (Atom *) = 0;
  virtual void reduce (bool strongly) = 0;
  virtual void unreduce () = 0;
  virtual void swapping (Follows *, Follows *) {};
  virtual void setup () = 0;
  virtual void print () {};

  RuleType type;
  bool visited;      // For use in backjumping only
};

class HeadRule : public Rule
{
public:
  HeadRule () { head = 0; };
  virtual ~HeadRule () {};

  Atom *head;        // The head of the rule
};

class BasicRule : public HeadRule
{
public:
  BasicRule ();
  virtual ~BasicRule ();

  void inactivate (Atom **);
  void inactivateInTrue (Atom **);
  void inactivateInFalse (Atom **);
  void fireInit ();
  void mightFireInitTrue (Atom **);
  void unInit ();
  void mightFireTrue (Atom **);
  void mightFireFalse (Atom **);
  void backChainTrue ();
  void backChainFalse ();
  void backtrackFromTrueInPos (Atom **);
  void backtrackFromFalseInPos (Atom **);
  void backtrackFromTrueInNeg (Atom **);
  void backtrackFromFalseInNeg (Atom **);
  void propagateFalse (Atom **);
  void propagateTrue (Atom **);
  void backtrackUpper (Atom **);

  bool isInactive ();
  bool isUpperActive ();
  void search (Atom *);
  void reduce (bool strongly);
  void unreduce ();
  void setup ();
  void print ();

  void mightFire ();

  long neg;          // The number of negative literals in body
                     // that are not in B- 
  long pos;          // The number of positive literals in body
		     // that are not in B+
  long pos2;         // The number of positive literals in body that
		     // are in the upper closure
  Atom **nbody;      // The negative literals in the body
  Atom **pbody;      // The positive literals in the body (allocated
		     // in nbody)
  Atom **nend;       // Sentinel of the nbody array
  Atom **pend;       // Sentinel of the pbody array
  Atom **end;        // The end of the body array
  long inactive;     // The number of positive literals in B- plus
		     // the number of negative literals in B+
};

// ConstraintRule implements the program
//           head :- A,B.
// where A is every instance of choosePos atoms of the positive
// literals and B is every instance of chooseNeg not-atoms of the
// negative literals.
// Example:
//            h :- a,b.
//            h :- a,c.
//            h :- b,c.
// has A = {a,b,c}, choosePos = 2, chooseNeg = 0, and head = h.
// With this rule one can implement
//         <n{...} :- ...
//         ... :- <n{...} ...
//   and   ... :- >n{...} ...
// Init:  pos = choosePos, neg = chooseNeg, inactivePos = choosePos -
// size of positive body, inactiveNeg = chooseNeg - size of negative
// body, pos2 = pos
class ConstraintRule : public HeadRule
{
public:
  ConstraintRule ();
  virtual ~ConstraintRule ();

  void inactivate (Atom **);
  void inactivateInTrue (Atom **);
  void inactivateInFalse (Atom **);
  void fireInit ();
  void mightFireInitTrue (Atom **);
  void unInit ();
  void mightFireTrue (Atom **);
  void mightFireFalse (Atom **);
  void backChainTrue ();
  void backChainFalse ();
  void backtrackFromTrueInPos (Atom **);
  void backtrackFromFalseInPos (Atom **);
  void backtrackFromTrueInNeg (Atom **);
  void backtrackFromFalseInNeg (Atom **);
  void propagateFalse (Atom **);
  void propagateTrue (Atom **);
  void backtrackUpper (Atom **);

  bool isInactive ();
  bool isUpperActive ();
  void search (Atom *);
  void reduce (bool strongly);
  void unreduce ();
  void setup ();
  void print ();

  Atom **nbody;      // The negative literals in the body
  Atom **pbody;      // The positive literals in the body (allocated
		     // in nbody)
  Atom **nend;       // Sentinel of the nbody array
  Atom **pend;       // Sentinel of the pbody array
  Atom **end;        // The end of the body array

  long pos;          // The number of positive literals in body that
		     // are not in B+
  long pos2;
  long neg;
  long inactivePos;  // The number of positive literals in B-
  long inactiveNeg;  // The number of negative literals in B+
};

class GenerateRule : public Rule
{
public:
  GenerateRule ();
  virtual ~GenerateRule ();

  void inactivate (Atom **);
  void inactivateInTrue (Atom **);
  void inactivateInFalse (Atom **);
  void fireInit ();
  void mightFireInitTrue (Atom **);
  void unInit ();
  void mightFire ();
  void mightFireTrue (Atom **);
  void mightFireFalse (Atom **);
  void backChainTrue ();
  void backChainFalse ();
  void backtrackFromTrueInPos (Atom **);
  void backtrackFromFalseInPos (Atom **);
  void backtrackFromTrueInNeg (Atom **);
  void backtrackFromFalseInNeg (Atom **);
  void propagateFalse (Atom **);
  void propagateTrue (Atom **);
  void backtrackUpper (Atom **);

  bool isInactive ();
  bool isUpperActive ();
  void search (Atom *);
  void reduce (bool strongly);
  void unreduce ();
  void setup ();
  void print ();

  Atom **head;       // The heads of the rule
  Atom **hend;       // Sentinel of the head array
  Atom **pbody;      // The atoms in the body (allocated in head)
  Atom **pend;       // Sentinel of the body array
  Atom **end;        // The end of the array

  long pos;
  long pos2;
  long neg;
  long inactivePos;  // The number of positive literals in B-
  long inactiveNeg;  // The number of negative literals (heads) in B+
};

class ChoiceRule : public Rule
{
public:
  ChoiceRule ();
  virtual ~ChoiceRule ();

  void inactivate (Atom **);
  void inactivateInTrue (Atom **);
  void inactivateInFalse (Atom **);
  void fireInit ();
  void mightFireInitTrue (Atom **);
  void unInit ();
  void mightFire ();
  void mightFireTrue (Atom **);
  void mightFireFalse (Atom **);
  void backChainTrue ();
  void backChainFalse ();
  void backtrackFromTrueInPos (Atom **);
  void backtrackFromFalseInPos (Atom **);
  void backtrackFromTrueInNeg (Atom **);
  void backtrackFromFalseInNeg (Atom **);
  void propagateFalse (Atom **);
  void propagateTrue (Atom **);
  void backtrackUpper (Atom **);

  bool isInactive ();
  bool isUpperActive ();
  void search (Atom *);
  void reduce (bool strongly);
  void unreduce ();
  void setup ();
  void print ();

  Atom **head;       // The heads of the rule
  Atom **hend;       // Sentinel of the head array
  long neg;          // The number of negative literals in body
                     // that are not in B- 
  long pos;          // The number of positive literals in body
		     // that are not in B+
  long pos2;         // The number of positive literals in body that
		     // are in the upper closure
  Atom **nbody;      // The negative literals in the body (allocated
                     // in head) 
  Atom **pbody;      // The positive literals in the body (allocated
		     // in head)
  Atom **nend;       // Sentinel of the nbody array
  Atom **pend;       // Sentinel of the pbody array
  Atom **end;        // The end of the body array
  long inactive;     // The number of positive literals in B- plus
		     // the number of negative literals in B+
};

class WeightRule : public HeadRule
{
public:
  WeightRule ();
  virtual ~WeightRule ();

  void change (Atom *);
  void inactivate (Atom **);
  void inactivateInTrue (Atom **);
  void inactivateInFalse (Atom **);
  void fireInit ();
  bool fired ();
  void mightFireInitTrue (Atom **);
  void mightFireInitFalse (Atom **);
  void unInit ();
  void mightFire (Atom **);
  void mightFireTrue (Atom **);
  void mightFireFalse (Atom **);
  void backChainTrue ();
  void backChainFalse ();
  void backtrackFromTrueInPos (Atom **);
  void backtrackFromFalseInPos (Atom **);
  void backtrackFromTrueInNeg (Atom **);
  void backtrackFromFalseInNeg (Atom **);
  void propagateFalse (Atom **);
  void propagateTrue (Atom **);
  void backtrackUpper (Atom **);
  void propagateFalseInNeg (Atom **);
  void propagateTrueInNeg (Atom **);
  void backtrackUpperInNeg (Atom **);
  void backtrack (Atom **);
  void upper_lower (Atom **);

  bool isInactive ();
  bool isUpperActive ();
  void search (Atom *);
  void reduce (bool strongly);
  void unreduce ();
  void initWeight (Weight);
  void setup ();
  void swap (long, long);
  void heap (long, long);
  bool smaller (long, long);
  void sort ();
  void swapping (Follows *, Follows *);

  void print ();

  Atom **bend;       // Sentinel of the body array
  Atom **body;       // The literals in the body
  Atom **end;        // The end of the array
  bool *positive;    // Is the literal an atom?
  Weight *weight;    // Local weights of atoms in body
  Weight maxweight;
  Weight minweight;
  Weight upper_max;   // Used in upper closure
  Weight upper_min;   // Used in upper closure
  Weight atmost;
  Weight atleast;
  Atom **upper;     // Position of the literal of largest absolute
  Atom **lower;     // value of weight not in B (atmost/atleast)
  Atom **upper_shadow; // What upper will be when queues are
  Atom **lower_shadow; // empty, reset to upper when backtracking
  Follows **reverse;
};

// This is really a metarule. It won't interact with the truth-values
// of the atoms.
class OptimizeRule : public Rule
{
public:
  OptimizeRule ();
  virtual ~OptimizeRule ();

  void setOptimum ();

  void inactivate (Atom **);
  void inactivateInTrue (Atom **);
  void inactivateInFalse (Atom **);
  void fireInit ();
  void mightFireInitTrue (Atom **);
  void unInit ();
  void mightFire (Atom **);
  void mightFireTrue (Atom **);
  void mightFireFalse (Atom **);
  void backChainTrue ();
  void backChainFalse ();
  void backtrackFromTrueInPos (Atom **);
  void backtrackFromFalseInPos (Atom **);
  void backtrackFromTrueInNeg (Atom **);
  void backtrackFromFalseInNeg (Atom **);
  void propagateFalse (Atom **);
  void propagateTrue (Atom **);
  void backtrackUpper (Atom **);

  bool isInactive ();
  bool isUpperActive ();
  void search (Atom *);
  void reduce (bool strongly);
  void unreduce ();
  void initWeight (Weight);
  void setup ();
  void print ();

  Atom **nbody;      // The negative literals in the body
  Atom **pbody;      // The positive literals in the body (allocated
		     // in nbody)
  Atom **nend;       // Sentinel of the nbody array
  Atom **pend;       // Sentinel of the pbody array
  Atom **end;        // The end of the body array
  Weight *weight;    // Local weights of atoms in nbody
  Weight maxweight;
  Weight minweight;
  Weight minoptimum;
  Weight maxoptimum;
  bool maximize;
  OptimizeRule *next;
};

#endif



