// Copyright 1997 Patrik Simons
// This software is provided as is, no warranty of any kind is given.
//
// File:   dl.cc
// Author: Patrik Simons
//
#include <iostream.h>
#include <iomanip.h>
#include <unistd.h>

#include "print.h"
#include "dl.h"

Dl::Dl ()
  : dcl (*this)
{
  nant = 0;
}

Dl::~Dl ()
{
  delete[] nant;
}

void
Dl::Init ()
{
  dcl.Init (number_of_atoms, number_of_rules, size_of_sigma - number_of_rules);
  nant = new Atom *[number_of_atoms];
  stack.Init (number_of_atoms);
  fail = 0;
  guesses = 0;
  conflict_found = 0;
  answer_number = 0;
  number_of_choice_points = 0;
  number_of_wrong_choices = 0;
  number_of_backjumps = 0;
  number_of_picked_atoms = 0;
  number_of_forced_atoms = 0;
  lasti = 0;
}

void 
Dl::nantInit ()
{
  nant_size = 0;
  Atom **a;
  for (long i = 0; i < number_of_rules; i++)
    for (a = dcl.rules[i].body; a != dcl.rules[i].endBody; a++)
      if (!(*a)->isnant)
	{
	  nant[nant_size++] = *a;
	  (*a)->isnant = true;
	}
  nanttop = nant_size;
  notcovered = nant_size;
}

inline void
Dl::removenant (long n)
{
  assert (nanttop);
  nanttop--;
  Atom *a = nant[nanttop];
  nant[nanttop] = nant[n];
  nant[n] = a;
}

inline void
Dl::addnant (Atom *a)
{
  while (nant[nanttop] != a)
    nanttop++;
  nanttop++;
  assert (nanttop <= nant_size);
}

inline void
Dl::resetDependency ()
{
  for (long i = 0; i < nanttop; i++)
    {
      nant[i]->posScore = -1;
      nant[i]->negScore = -1;
      nant[i]->dependsonTrue = false;
      nant[i]->dependsonFalse = false;
    }
}

int
Dl::setToBTrue (Atom *a)
{
  if (a->Bneg)
    {
      conflict_found = a;
      return 1;
    }
  else
    {
      if (a->isnant)
	{
	  notcovered--;
	  score++;
	  a->dependsonTrue = true;
	}
      dcl.setBTrue (a);
      stack.push (a);
      return 0;
    }
}

int
Dl::setToBFalse (Atom *a)
{
  if (a->Bpos)
    {
      conflict_found = a;
      return 1;
    }
  else
    {
      if (a->isnant)
	{
	  notcovered--;
	  score++;
	  a->dependsonFalse = true;
	}
      dcl.setBFalse (a);
      stack.push (a);
      return 0;
    }
}

void
Dl::pick ()
{
  number_of_picked_atoms++;
  number_of_choice_points++;
  long least_n = 0x7fffffff;
  Atom *picked = 0;
  Atom *alt = 0;
  long pn = 0, an = 0;
  for (long i = 0; i < nanttop && least_n > 1; i++)
    {
      Atom *a = nant[i];
      if (a->Bpos || a->Bneg)
	{
	  removenant (i);
	  i--;
	}
      else
	{
	  alt = a;
	  an = i;
	  for (Rule **f = a->neg; f != a->endNeg; f++)
	    if ((*f)->pos == 0 && !(*f)->inactive
		&& (*f)->neg < least_n && (*f)->head->Bpos == false)
	      {
		picked = a;
		pn = i;
		least_n = (*f)->neg;
		if (least_n == 1)
		  break;
	      }
	}
    }
  if (!picked)
    {
      picked = alt;
      pn = an;
    }
  PRINT_PICK (cout << "Picked " << picked->name << endl);
  stack.push (picked);
  picked->guess = true;
  guesses++;
  removenant (pn);
  notcovered--;
  dcl.setBFalse (picked);
  cnflct = 0;
}

inline void
Dl::expand ()
{
  dcl.dcl ();
}

inline int
Dl::conflict ()
{
  if (conflict_found)
    {
      PRINT_CONFLICT (cout << "Conflict: " <<
		      conflict_found->name << endl);
      dcl.truequeue.reset ();
      dcl.etruequeue.reset ();
      dcl.efalsequeue.reset ();
      dcl.falsequeue.reset ();
      conflict_found = 0;
      return 1;
    }
  else
    return 0;
}

void
Dl::setup ()
{
  dcl.setup ();
  if (conflict_found) // Can't use conflict() as this removes the conflict.
    return;
  expand ();  // Compute well-founded model
  dcl.reduceStrongly ();  // Reduce the program
  // Initialize literals chosen to be in the stable model / full set.
  for (Atom *a = dcl.atoms; a != dcl.endAtom; a++)
    {
      if (a->dependsonFalse)          // Negative literal
	{
	  a->dependsonFalse = false;
	  if (a->Bneg == false)
	    if (setToBFalse (a))
	      return;
	}
      if (a->dependsonTrue)           // Positive literal
	{
	  a->dependsonTrue = false;
	  if (a->Bpos == false)
	    if (setToBTrue (a))
	      return;
	}
    }
  expand ();
  if (conflict_found)
    return;
  dcl.reduceWeakly ();
  dcl.improve ();
  stack.reset ();
  level = 0;
  cnflct = 0;
}

inline int 
Dl::covered ()
{
  return notcovered == 0;
}

Atom *
Dl::unwind ()
{
  Atom *a = stack.pop ();
  while (a->guess == false)
    {
      PRINT_STACK (a->backtracked = false; a->forced = false);
      if (a->Bpos)
	{
	  dcl.backtrackFromBTrue (a);
	  if (a->isnant)
	    notcovered++;
	}
      else if (a->Bneg)
	{
	  dcl.backtrackFromBFalse (a);
	  if (a->isnant)
	    notcovered++;
	}
      a = stack.pop ();
    }
  a->guess = false;
  guesses--;
  return a;
}

Atom *
Dl::backtrack ()
{
  if (guesses == 0)
    {
      fail = 1;
      return 0;
    }
  Atom *a = unwind ();
  PRINT_BACKTRACK (cout << "Backtracking: " << a->name << endl);
  if (a->Bneg)
    {
      dcl.backtrackFromBFalse (a);
      dcl.setBTrue (a);
      stack.push (a);
      PRINT_STACK (a->backtracked = true);
    }
  else
    {
      dcl.backtrackFromBTrue (a);
      dcl.setBFalse (a);
      stack.push (a);
      PRINT_STACK (a->backtracked = true);
    }
  return a;
}

Atom *
Dl::backjump ()
{
  for (;;) 
    {
      if (guesses == 0)
	{
	  fail = 1;
	  return 0;
	}
      Atom *a = unwind ();
      PRINT_BACKTRACK (cout << "Backtracking: " << a->name << endl);
      bool b = a->Bneg;
      if (a->Bneg)
	{
	  dcl.backtrackFromBFalse (a);
	  if (a->isnant)
	    notcovered++;
	  PRINT_STACK (a->backtracked = false; a->forced = false);
	}
      else
	{
	  dcl.backtrackFromBTrue (a);
	  if (a->isnant)
	    notcovered++;
	  PRINT_STACK (a->backtracked = false; a->forced = false);
	}
      if (cnflct == 0 || guesses < level || dcl.path (cnflct,a))
	{
	  if (guesses < level)
	    level = guesses;
	  if (b)
	    dcl.setBTrue (a);
	  else
	    dcl.setBFalse (a);
	  stack.push (a);
	  if (a->isnant)
	    notcovered--;
	  cnflct = a;
	  PRINT_STACK (a->backtracked = true);
	  return a;
	}
      number_of_backjumps++;
    }
}

inline int
Dl::testPos (Atom *a)
{
  score = 0;
  stack.push (a);
  a->guess = true;
  guesses++;
  dcl.setBTrue (a);
  number_of_picked_atoms++;
  expand ();
  if (conflict ())
    {
      // Backtrack puts the atom onto the stack.
      number_of_forced_atoms++;
      notcovered--;
      backtrack ();
      cnflct = a;
      PRINT_STACK (a->forced = true);
      return 1;
    }
  unwind ();
  dcl.backtrackFromBTrue (a);
  return 0;
}

inline int
Dl::testNeg (Atom *a)
{
  score = 0;
  stack.push (a);
  a->guess = true;
  guesses++;
  dcl.setBFalse (a);
  number_of_picked_atoms++;
  expand ();
  if (conflict ())
    {
      // Backtrack puts the atom onto the stack.
      number_of_forced_atoms++;
      notcovered--;
      backtrack ();
      cnflct = a;
      PRINT_STACK (a->forced = true);
      return 1;
    }
  unwind ();
  dcl.backtrackFromBFalse (a);
  return 0;
}

inline void
Dl::testScore (long pos, long neg, long i, long &hiscore1,
	      long &hiscore2, long &hii, int &positive)
{
  long mn, mx;
  if (neg < pos)
    {
      mn = neg;
      mx = pos;
    }
  else
    {
      mn = pos;
      mx = neg;
    }
  if (mn > hiscore1)
    {
      hiscore1 = mn;
      hiscore2 = mx;
      hii = i;
      if (mn == pos)
	positive = 0;
      else
	positive = 1;
    }
  else if (mn == hiscore1)
    if (mx >= hiscore2)
      {
	hiscore2 = mx;
	hii = i;
	if (mn == pos)
	  positive = 0;
	else
	  positive = 1;
      }
}

//
// Choose a literal that gives rise to a conflict. 
// If no such literal exists we choose the literal 
// that brings the most NAnt atoms into the closures.
//
void
Dl::lookahead ()
{
  long hiscore1 = 0;
  long hiscore2 = 0;
  int positive = 0;
  long hii = 0;
  long i;
  Atom *a;
  
  resetDependency ();
  bool firstPass = true;
  for (i = lasti; ;)
    {
      if (firstPass)
	{
	  if (i >= nanttop)
	    {
	      firstPass = false;
	      i = 0;
	      continue;
	    }
	}
      else if (i >= nanttop || i >= lasti)
	break;
      a = nant[i];
      if (a->Bpos || a->Bneg)
	{
	  removenant (i);
	  continue;
	}
      if (a->dependsonFalse == false)
	if (testNeg (a))
	  {
	    lasti = i;
	    return;
	  }
	else
	  a->negScore = score;
      if (a->dependsonTrue == false)
	if (testPos (a))
	  {
	    lasti = i;
	    return;
	  }
	else
	  a->posScore = score;
      if (a->posScore != -1 && a->negScore != -1)
	testScore (a->posScore, a->negScore, i, hiscore1, hiscore2, hii,
		   positive);
      i++;
    }
  lasti = 0;
  for (i = 0; i < nanttop; i++)
    {
      a = nant[i];
      if (a->negScore >= 0 && a->negScore <= hiscore1)
	continue;
      if (a->posScore >= 0 && a->posScore <= hiscore1)
	continue;
      if (a->negScore == -1)
	{
	  testNeg (a);
	  a->negScore = score;
	}
      if (a->negScore <= hiscore1)
	continue;
      if (a->posScore == -1)
	{
	  testPos (a);
	  a->posScore = score;
	}
      testScore (a->posScore, a->negScore, i, hiscore1, hiscore2, hii,
		 positive);
    }
  a = nant[hii];
  PRINT_PICK (if (positive)
	      cout << "Chose " << a->name << endl;
	      else
	      cout << "Chose not " << a->name << endl);
  number_of_picked_atoms++;
  number_of_choice_points++;
  stack.push (a);
  a->guess = true;
  guesses++;
  removenant (hii);
  cnflct = 0;
  notcovered--;
  if (positive)
    dcl.setBTrue (a);
  else
    dcl.setBFalse (a);
}

int
Dl::smodels (int dolookahead, int backjumping)
{
  Atom *a;
  setup ();
  if (conflict ())
    return 0;
  do
    {
      PRINT_DCL (cout << "Smodels call" << endl;
		 dcl.print ());
      PRINT_BF (print ());
      expand ();
      PRINT_BF (cout << "Expand" << endl);
      PRINT_DCL (dcl.print ());
      PRINT_BF (print ());
      PRINT_PROGRAM(printProgram ());
      PRINT_STACK (printStack ());
      if (conflict ())
	{
	  if (backjumping)
	    a = backjump ();
	  else
	    a = backtrack ();
	  if (a)
	    addnant (a);
	  number_of_wrong_choices++;
	}
      else if (covered ())
	{
	  answer_number++;
	  level = guesses;
	  cout << "Answer: " << answer_number << endl;
	  printAnswer ();
	  if (max_models && answer_number >= max_models)
	    return 1;
	  else
	    {
	      if (backjumping)
		a = backjump ();
	      else
		a = backtrack ();
	      if (a)
		addnant (a);
	      number_of_wrong_choices++;
	    }
	} 
      else if (dolookahead)
	lookahead ();
      else
	pick ();
    } 
  while (!fail);
  number_of_wrong_choices--;
  return 0;
}

int
Dl::wellfounded ()
{
  setup ();
  if (conflict ())
    return 0;
  cout << "Well-founded model: " << endl;
  cout << "Positive part: ";
  Atom *a;
  for (a = dcl.atoms; a != dcl.endAtom; a++)
    if (a->Bpos)
      cout << a->name << ' ';
  cout << endl << "Negative part: ";
  for (a = dcl.atoms; a != dcl.endAtom; a++)
    if (a->Bneg)
      cout << a->name << ' ';
  cout << endl;
  return 1;
}

void
Dl::Calculate (int w, int lookahead, int backjumping)
{
  int r;
  if (w)
    r = wellfounded ();
  else
    r = smodels (lookahead, backjumping);
  if (r)
    cout << "True" << endl;
  else
    cout << "False" << endl;
  timer.stop ();
  cout << "Duration: " << timer.print () << endl;
  cout << "Number of choice points: " << number_of_choice_points <<
    endl;
  cout << "Number of wrong choices: " << number_of_wrong_choices << endl;
  cout << "Number of atoms: " << number_of_atoms << endl;
  cout << "Number of rules: " << number_of_rules << endl;
  if (lookahead)
    {
      cout << "Number of picked atoms: " << number_of_picked_atoms << endl;
      cout << "Number of forced atoms: " << number_of_forced_atoms << endl;
    }
  if (backjumping)
    cout << "Number of backjumps: " << number_of_backjumps << endl;
}

void
Dl::print ()
{
  Atom *a;
  cout << "Body: ";
  for (a = dcl.atoms; a != dcl.endAtom; a++)
    if (a->Bpos)
      cout << a->name << ' ';
  cout << endl << "NBody: ";
  for (a = dcl.atoms; a != dcl.endAtom; a++)
    if (a->Bneg)
      cout << a->name << ' ';
  cout << endl << "Pick: ";
  Atom **b;
  for (b = stack.stack; b != stack.stack+stack.top; b++)
    if ((*b)->guess)
      if((*b)->Bneg)
	cout << "not " << (*b)->name << ' ';
      else
	cout << (*b)->name << ' ';
  cout << endl;
}

void
Dl::printAnswer ()
{
  // Prints the stable model.
  Atom *a;
  cout << "Stable Model: ";
  for (a = dcl.atoms; a != dcl.endAtom; a++)
    if (a->Bpos)
      cout << a->name << ' ';
  cout << endl << "Full set: ";
  for (a = dcl.atoms; a != dcl.endAtom; a++)
    if (a->Bneg && a->isnant)
      cout << a->name << ' ';
  cout << endl;
}

void
Dl::printProgram ()
{
  cout << "Program:" << endl;
  for (Rule *r = dcl.rules; r != dcl.endRule; r++)
    {
      if (r->inactive)
      	continue;
      //      if (r->head->Bpos)
      //       	continue;
      //      if (r->head->Bneg)
      //      	continue;
      printRule (r);
    }
}

void
Dl::printRule (Rule *r)
{
  cout << r->head->name << " :- ";
  Atom **a;
  int comma = 0;
  for (a = r->firstPos; a != r->endBody; a++)
    if ((*a)->Bpos == false)
      {
	if (comma)
	  cout << ", ";
	cout << (*a)->name;
	comma = 1;
      }
  for (a = r->body; a != r->firstPos; a++)
    if ((*a)->Bneg == false)
      {
	if (comma)
	  cout << ", ";
	cout << "not " << (*a)->name;
	comma = 1;
      }
  cout << '.' << endl;
}

void
Dl::printStack ()
{
  long i;
  cout << "\x1b[1;1fStack: ";
  for (i = 0; i < stack.top; i++)
    {
      Atom *a = stack.stack[i];
      if (a->forced)
	cout << "\x1b[31m";
      else if (a->backtracked)
	cout << "\x1b[32m";
      else if (a->guess)
	cout << "\x1b[34m";
      if (a->Bneg)
	cout << "not " << a->name;
      else
	cout << a->name;
      cout << "\x1b[0m ";
    }
  cout << "\x1b[0J" << endl;
  //  sleep(1);
}
