// Copyright 1997 Patrik Simons
// This software is provided as is, no warranty of any kind is given.
//
// File:   dcl.cc
// Author: Patrik Simons
//
#include <iostream.h>

#include "dl.h"


Dcl::Dcl (Dl &d)
  : dl (d)
{
  tmpfalse = 0;
  tmpfalsetop = 0;
  atoms = 0;
  rules = 0;
  atomsSize = 0;
  rulesSize = 0;
  atom_holder = 0;
  rule_holder = 0;
}

Dcl::~Dcl ()
{
  delete[] tmpfalse;
  delete[] atoms;
  delete[] rules;
  delete[] atom_holder;
  delete[] rule_holder;
}

void
Dcl::Init (long numatoms, long numrules, long bodysize)
{
  atomsSize = numatoms;
  rulesSize = numrules;

  atom_holder = new Atom *[bodysize];
  rule_holder = new Rule *[bodysize+rulesSize];
  atoms = new Atom[atomsSize];
  rules = new Rule[rulesSize];
  endAtom = atoms+atomsSize;
  endRule = rules+rulesSize;

  truequeue.Init (rulesSize);
  etruequeue.Init (bodysize+rulesSize);    // Was rules, but is now enlarged
  efalsequeue.Init (bodysize+rulesSize);   // since we are doing backchaining.
  falsequeue.Init (rulesSize);
  tmpfalse = new Atom *[atomsSize];
  tmpfalsetop = 0;
}

void
Dcl::setup ()
{
  for (Rule *r = rules; r != endRule; r++)
    {
      if (r->pos == 0 && r->neg == 0)
	{
	  etruequeue.push (r->head);
	  truequeue.push (r->head);
	}
      else if (r->pos2 == 0)
	truequeue.push (r->head);
    }
  // Find atoms that are true.
  Atom *a;
  while (!truequeue.empty ())
    {
      a = truequeue.pop ();
      if (a->closure == false)
	setTrue (a);
    }
  // Set not true atoms to explicitly false.
  for (a = atoms; a != endAtom; a++)
    if (a->closure == false)
      if (dl.setToBFalse (a))
	return;
}

void
Dcl::dcl ()
{
  Atom *a;
  for (;;)
    {
      // Propagate explicit truth. (Fitting Semantics)
      if (!etruequeue.empty ())
	{
	  a = etruequeue.pop ();
	  if (a->Bpos == false)
	    {
	      // dl.setToBtrue calls dcl->setBTrue if no conflict
	      // is found, otherwise returns 1.
	      if (dl.setToBTrue (a))
		return;
	    }
	}
      // Propagate explicit falsehood. (Fitting Semantics)
      else if (!efalsequeue.empty ())
	{
	  a = efalsequeue.pop ();
	  // dl.setToBFalse calls dcl->setBFalse if no conflict
	  // is found, otherwise returns 1.
	  if (a->Bneg == false)
	    if (dl.setToBFalse (a))
	      return;
	}
      // Propagate falsehood.
      // Find conflicts in order to guarantee soundness of smodels.
      // This part prevents dcl from being incrementally linear.
      else if (!falsequeue.empty ())
	{
	  if (propagateFalse ())
	    return;
	}
      else
	return;
    }
}

int
Dcl::propagateFalse ()
{
  Atom *a;
  Rule **f;
  long i;
  tmpfalsetop = 0;
  // Propagate falsehood.
  while (!falsequeue.empty ())
    {
      a = falsequeue.pop ();
      // If a->closure is false, then the descendants of a are
      // in falsequeue.
      if (a->closure == true && a->Bneg == false)
	{
	  for (f = a->pos; f != a->endPos2; f++)
	    {
	      (*f)->pos2++;
	      if ((*f)->pos2 == 1 && !(*f)->inactive &&
		  (*f)->head->closure != false)
		falsequeue.push ((*f)->head);
	    }
	  a->closure = false;
	  tmpfalse[tmpfalsetop] = a;
	  tmpfalsetop++;
	}
    }
  // Some atoms now false should be true.
  for (i = 0; i < tmpfalsetop; i++)
    if (tmpfalse[i]->headof)
      for (f = tmpfalse[i]->head; f != tmpfalse[i]->endHead; f++)
	if ((*f)->pos2 == 0 && !(*f)->inactive)
	  {
	    truequeue.push (tmpfalse[i]);
	    break;
	  }
  // Propagate truth to find them all.
  while (!truequeue.empty ())
    {
      a = truequeue.pop ();
      if (a->closure == false && a->Bneg == false)
	{
	  for (f = a->pos; f != a->endPos2; f++)
	    {
	      (*f)->pos2--;
	      if ((*f)->pos2 == 0 && !(*f)->inactive &&
		  (*f)->head->closure == false)
		truequeue.push ((*f)->head);
	    }
	  a->closure = true;
	}
    }
  // Those atoms now false can be set to explicitly false.
  for (i = 0; i < tmpfalsetop; i++)
    {
      a = tmpfalse[i];
      if (a->closure == false)
	if (dl.setToBFalse (a))
	  {
	    // Conflict found, we reset the f->pos2 variables that are
	    // not reset while backtracking.
	    for (; i < tmpfalsetop; i++)
	      {
		a = tmpfalse[i];
		if (a->closure == false)
		  for (f = a->pos; f != a->endPos2; f++)
		    (*f)->pos2--;
		a->closure = true;
	      }
	    return 1;
	  }
    }
  return 0;
}

inline void
Dcl::inactivate (Rule *f, Atom *a)
{
  f->inactive++;
  if (f->inactive == 1)
    {
      Atom *b = f->head;
      b->headof--;
      if (b->headof && b->Bneg == false)
	falsequeue.push (b);
      if (b->headof == 0 && b->Bneg == false)
	efalsequeue.push (b);
      else if (b->headof == 1 && b->Bpos && a != b)
	backchainTrue (b);   // Prevent pathological etruequeue overflow
    }
}

inline void
Dcl::mightFireRule (Rule *f)
{
  if (f->pos == 0)
    {
      if (f->neg == 0 && f->head->Bpos == false)
	etruequeue.push (f->head);
      else if (f->neg == 1 && f->head->Bneg)
	backchainRuleFalse (f);
    }
  else if (f->pos == 1 && f->neg == 0 && f->head->Bneg)
    backchainRuleFalse (f);
}

//
// Note that since backchain depends on the truth values of the
// atoms in the body of the rules, we must set a->B first.
// To keep everything safe we also handle inactivation before 
// we fire any rules.
//
void
Dcl::setBTrue (Atom *a)
{
  Rule **f;
  a->Bpos = true;
  for (f = a->neg; f != a->endNeg; f++)
    inactivate (*f, a);
  for (f = a->pos; f != a->endPos; f++)
    {
      (*f)->pos--;
      mightFireRule (*f);
    }
  // This atom could have been forced.
  if (a->headof == 1)
    backchainTrue (a);
}

void
Dcl::setBFalse (Atom *a)
{
  Rule **f;
  a->Bneg = true;
  for (f = a->pos; f != a->endPos; f++)
    {
      if (a->closure == true)
	(*f)->pos2++;
      inactivate (*f, 0);
    }
  a->closure = false;
  for (f = a->neg; f != a->endNeg; f++)
    {
      (*f)->neg--;
      mightFireRule (*f);
    }
  // This atom could have been forced.
  if (a->headof)        // There are active rules.
    backchainFalse (a); // Might backchain already backchained rules
			// in mightFireRule, doesn't matter as no
			// efalsequeue overflow will happen.
}

void
Dcl::setTrue (Atom *a)
{
  for (Rule **f = a->pos; f != a->endPos; f++)
    {
      (*f)->pos2--;
      if ((*f)->pos2 == 0 && !(*f)->inactive)
	truequeue.push ((*f)->head);
    }
  a->closure = true;
}

void
Dcl::backtrackFromBTrue (Atom *a)
{
  Rule **f;
  for (f = a->pos; f != a->endPos; f++)
    (*f)->pos++;
  for (f = a->neg; f != a->endNeg; f++)
    {
      (*f)->inactive--;
      if ((*f)->inactive == 0)
	(*f)->head->headof++;
    }
  a->Bpos = false;
  a->closure = true; // Not necessary
}

void
Dcl::backtrackFromBFalse (Atom *a)
{
  Rule **f;
  for (f = a->neg; f != a->endNeg; f++)
    (*f)->neg++;
  for (f = a->pos; f != a->endPos; f++)
    {
      (*f)->pos2--;
      (*f)->inactive--;
      if ((*f)->inactive == 0)
	(*f)->head->headof++;
    }
  a->Bneg = false;
  a->closure = true;
}

void 
Dcl::backchainTrue (Atom *a)
{
  Rule **f;
  Atom **b;
  // Find the only active rule.
  for (f = a->head; f != a->endHead; f++)
    if (!(*f)->inactive)
      break;
  if ((*f)->neg)
    for (b = (*f)->body; b != (*f)->firstPos; b++)
      if ((*b)->Bneg == false)
	efalsequeue.push (*b);
  if ((*f)->pos)
    for (b = (*f)->firstPos; b != (*f)->endBody; b++)
      if ((*b)->Bpos == false)
	etruequeue.push (*b);
}

void 
Dcl::backchainFalse (Atom *a)
{
  for (Rule **f = a->head; f != a->head; f++)
    backchainRuleFalse (*f);
}

void 
Dcl::backchainRuleFalse (Rule *f)
{
  Atom **b;
  if (f->inactive)
    return;
  if (f->pos == 0 && f->neg == 1)
    {
      for (b = f->body; b != f->firstPos; b++)
	if ((*b)->Bneg == false)
	  {
	    etruequeue.push (*b);
	    break;
	  }
    }
  else if (f->pos == 1 && f->neg == 0)
    for (b = f->firstPos; b != f->endBody; b++)
      if ((*b)->Bpos == false)
	{
	  efalsequeue.push (*b);
	  break;
	}
}

inline void
Dcl::searchRule (Rule *f)
{
  if (!f->head->visited && f->head->Bneg == false)
    {
      truequeue.push (f->head);
      f->head->visited = true;
    }
  for (Atom **a = f->body; a != f->endBody; a++)
    if (!(*a)->visited && (*a)->Bneg == false)
      {
	truequeue.push (*a);
	(*a)->visited = true;
      }
  f->visited = true;
}

int
Dcl::path (Atom *a, Atom *b)
{
  if (a == b)
    return 1;
  Rule **f;
  Atom *c;
  long i;
  for (i = 0; i < rulesSize; i++)
    rules[i].visited = false;
  for (i = 0; i < atomsSize; i++)
    atoms[i].visited = false;
  truequeue.push (a);
  a->visited = true;
  while (!truequeue.empty ())
    {
      c = truequeue.pop ();
      if (c == b)
	{
	  truequeue.reset ();
	  return 1;
	}
      for (f = a->head; f != a->endHead; f++)
	if (!(*f)->visited && !(*f)->inactive)
	  searchRule (*f);
      for (f = a->pos; f != a->endPos; f++)
	if (!(*f)->visited && !(*f)->inactive)
	  searchRule (*f);
      for (f = a->neg; f != a->endNeg; f++)
	if (!(*f)->visited && !(*f)->inactive )
	  searchRule (*f);
    }
  return 0;
}

void
Dcl::reduceStrongly ()
{
  long i;
  Rule **r;
  Rule **t;
  for (i = 0; i < atomsSize; i++)
    {
      t = atoms[i].head;
      for (r = atoms[i].head; r != atoms[i].endHead; r++)
	{
	  *t = *r;
	  if (!(*t)->inactive)
	    t++;
	}
      atoms[i].endHead = t;
      t = atoms[i].pos;
      for (r = atoms[i].pos; r != atoms[i].endPos; r++)
	{
	  *t = *r;
	  if (!(*t)->inactive && atoms[i].Bpos == false)
	    t++;
	}
      atoms[i].endPos = t;
      atoms[i].endPos2 = t;
      t = atoms[i].neg;
      for (r = atoms[i].neg; r != atoms[i].endNeg; r++)
	{
	  *t = *r;
	  if (!(*t)->inactive && atoms[i].Bneg == false)
	    t++;
	}
      atoms[i].endNeg = t;
    }
  Atom **a;
  Atom **b;
  for (i = 0; i < rulesSize; i++)
    {
      if (rules[i].inactive)
	continue;   // Nothing points at this rule now
      Atom **n = rules[i].firstPos;
      b = rules[i].body;
      for (a = rules[i].body; a != rules[i].firstPos; a++)
	{
	  *b = *a;
	  if ((*b)->Bpos == false && (*b)->Bneg == false)
	    b++;
	}
      rules[i].firstPos = b;
      for (a = n; a != rules[i].endBody; a++)
	{
	  *b = *a;
	  if ((*b)->Bpos == false && (*b)->Bneg == false)
	    b++;
	}
      rules[i].endBody = b;
    }
}

void
Dcl::reduceWeakly ()
{
  long i;
  Rule **r;
  Rule **t;
  for (i = 0; i < atomsSize; i++)
    {
      t = atoms[i].head;
      for (r = atoms[i].head; r != atoms[i].endHead; r++)
	{
	  *t = *r;
	  if (!(*t)->inactive)
	    t++;
	}
      atoms[i].endHead = t;
      t = atoms[i].pos;
      for (r = atoms[i].pos; r != atoms[i].endPos; r++)
	{
	  *t = *r;
	  if (!(*t)->inactive)
	    t++;
	}
      atoms[i].endPos = t;
      atoms[i].endPos2 = t;
      t = atoms[i].neg;
      for (r = atoms[i].neg; r != atoms[i].endNeg; r++)
	{
	  *t = *r;
	  if (!(*t)->inactive && atoms[i].Bneg == false)
	    t++;
	}
      atoms[i].endNeg = t;
    }
  Atom **a;
  Atom **b;
  for (i = 0; i < rulesSize; i++)
    {
      if (rules[i].inactive)
	continue;
      Atom **n = rules[i].firstPos;
      b = rules[i].body;
      for (a = rules[i].body; a != rules[i].firstPos; a++)
	{
	  *b = *a;
	  if ((*b)->Bpos == false && (*b)->Bneg == false)
	    b++;
	}
      rules[i].firstPos = b;
      for (a = n; a != rules[i].endBody; a++)
	*b++ = *a;  // Can't remove atoms in B+
      rules[i].endBody = b;
    }
}

void
Dcl::print ()
{
  long i;
  cout << "bdcl:" << endl << "True:";
  for (i = 0; i < atomsSize; i++) 
    if (atoms[i].closure == true)
      cout << ' ' << atoms[i].name;
  cout << endl << "BTrue:";
  for (i = 0; i < atomsSize; i++) 
    if (atoms[i].Bpos)
      cout << ' ' << atoms[i].name;
  cout << endl << "BFalse:";
  for (i = 0; i < atomsSize; i++) 
    if (atoms[i].Bneg)
      cout << ' ' << atoms[i].name;
  cout << endl;
}
