// Copyright 1998 by Patrik Simons
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston,
// MA 02111-1307, USA.
//
// Patrik.Simons@hut.fi
#include <iostream.h>
#include <string.h>
#include <stdlib.h>
#include <float.h>
#include <limits.h>
#include "parser.h"
#include "tokens.h"

extern int yylex ();
extern char *yytext;
extern int yyleng;
extern long linenumber;
extern int computestatement;

Parser::Parser (ostream &o)
  : out (o)
{
  havetoken = false;
  token = 0;
  maxmodels = 1;
  lastid = 0;
  fail = 0;
}

Parser::~Parser ()
{
}

inline int
Parser::gettoken ()
{
  if (havetoken) {
    havetoken = false;
    return token;
  }
  token = yylex ();
  return token;
}

unsigned long
Parser::getnumber ()
{
  return strtoul (yytext, 0, 0);
}

void
Parser::getWeight (Atom *a)
{
  a->positive_weight = true;
# ifdef USEDOUBLE
  Weight w = strtod (yytext, 0);
  if (w < 0)
    {
      a->positive = false;
      w = -w;
    }
  return w;
# else
  char *text = yytext;
  if (*text == '-')
    {
      text++;
      a->positive_weight = false;
    }
  a->weight = strtoul (text, 0, 0);
# endif
}

Weight
Parser::getWeight ()
{
# ifdef USEDOUBLE
  Weight w = strtod (yytext, 0);
  if (w < 0)
    w = -w;
  return w;
# else
  char *text = yytext;
  if (*text == '-')
    text++;
  return strtoul (text, 0, 0);
# endif
}

int
Parser::program ()
{
  for (;;)
    {
      switch (gettoken ())
	{
	case COMPUTE:
	  if (compute () == Error)
	    return Error;
	  break;
	case MAXIMIZE:
	  if (optimize (true) == Error)
	    return Error;
	  break;
	case MINIMIZE:
	  if (optimize (false) == Error)
	    return Error;
	  break;
	case 0:
	  return Ok;  // End of file.
	default:
	  havetoken = true;
	  if (statement () == Error)
	    return Error;
	  break;
	}
    }
  return Ok;
}

void
Parser::error (char *s)
{
  cerr << s
       << " at line "
       << linenumber << endl;
  print_line ();
}

bool
Parser::isWeight (int t)
{
# ifdef USEDOUBLE
  return t == POSNUMBER || t == NEGNUMBER || t == DOUBLE;
# else
  return t == POSNUMBER || t == NEGNUMBER;
# endif
}

bool
Parser::isParameter (int t)
{
  return t == ATOM || t == POSNUMBER || t == NEGNUMBER || t == PARAMETER;
}

int
Parser::statement ()
{
  int token = gettoken ();
  havetoken = true;
  if (token == POSNUMBER)
    return generate ();
  if (token == LBRACE)
    return choice ();
  Atom *a = atom ();
  if (!a)
    {
      error ("I expected an atom as head or weight");
      return Error;
    }
  switch (gettoken ())
    {
    case RULE_ARROW:
    case RULE_END:
      havetoken = true;
      return rule (a);
    case EQUAL:
      token = gettoken ();
      if (!isWeight (token))
	{
	  error ("I expected a weight");
	  return Error;
	}
      getWeight (a);
      break;
    }
  return Ok;
}

int
Parser::generate ()
{
  gettoken ();
  r.clear ();
  r.type = GENERATERULE;
  r.chooseHead = getnumber ();
  if (braced (&r.head, true) == Error)
    {
      error ("The error is in the head of the rule");
      return Error;
    }
  int h = r.head.size ();
  if (h < 2)
    {
      error ("I expected at least two atoms in the head");
      return Error;
    }
  if (r.chooseHead < 1 || r.chooseHead >= h)
    {
      error ("Choose in head out of bounds");
      return Error;
    }
  switch (gettoken ())
    {
    case RULE_ARROW:
      if (array (&r.pbody, true) == Error)
	return Error;
      if (gettoken () != RULE_END)
	{
	  error ("I expected a `.'");
	  return Error;
	}
      break;
    case RULE_END:
      break;
    default:
      error ("I expected a `:-' or a `.'");
      return Error;
    }
  r.print (out);
  return Ok;
}

int
Parser::choice ()
{
  r.clear ();
  r.type = CHOICERULE;
  if (braced (&r.head, true) == Error)
    {
      error ("The error is in the head of the rule");
      return Error;
    }
  int h = r.head.size ();
  if (h < 1)
    {
      error ("I expected at least one atoms in the head");
      return Error;
    }
  return rule (0);
}

int
Parser::rule (Atom *a)
{
  if (a)
    {
      r.clear ();
      r.type = BASICRULE;
      r.head.push (a);
    }
  bool ret = Ok;
  switch (gettoken ())
    {
    case RULE_ARROW:
      if (gettoken () == POSNUMBER)
	{
	  ret = constraint (&r);
	  break;
	}
      else
	havetoken = true;
      if (token == LBRACE)
	{
	  if (braced (&r) == Error)
	    return Error;
	  ret = weight (&r);
	  break;
	}
      if (body (&r) == Error)
	return Error;
      if (gettoken () != RULE_END)
	{
	  error ("I expected a `.'");
	  return Error;
	}
      break;
    case RULE_END:
      break;
    default:
      error ("I expected a `:-' or a `.'");
      return Error;
    }
  if (ret == Ok)
    r.print (out);
  return ret;
}

int
Parser::constraint (Rule *r)
{
  r->type = CONSTRAINTRULE;
  r->chooseBody = getnumber ();
  if (r->chooseBody <= 0)
    {
      error ("I expected positive number as constraint");
      return Error;
    }
  if (braced (r) == Error)
    return Error;
  long ps = r->pbody.size ();
  long ns = r->nbody.size ();
  if (ns+ps == r->chooseBody)  // Optimization
    r->type = BASICRULE;
  else if (ns+ps == 0)
    {
      error ("I expected some literals in constraint");
      return Error;
    }
  if (gettoken () != RULE_END)
    {
      error ("I expected a `.'");
      return Error;
    }
  return Ok;
}

void
Parser::change_atmost_to_atleast (Rule *r)
{
  r->swap_pos_neg ();
  r->atleast = 0;
  for (int i = 0; i < r->pbody.top; i++)
    r->atleast += r->pbody.array[i].weight;
  for (int i = 0; i < r->nbody.top; i++)
    r->atleast += r->nbody.array[i].weight;
  if (r->atmost > r->atleast)
    r->atleast = 0;
  else
    r->atleast = r->atleast - r->atmost;
}

void
Parser::split_weight_rule (Rule *r)
{
  if (r->type == NORULE)
    return;
# ifdef USEDOUBLE
  Weight w = DBL_MAX;
# else
  Weight w = ULONG_MAX;
# endif
  if (r->atleast == 0)
    change_atmost_to_atleast (r);
  else if (w != r->atmost)
    {
      Atom *h1 = newAtom (0);
      Atom *h2 = newAtom (0);
      tmp.type = BASICRULE;
      tmp.head.push (r->head.array[0].atom);
      tmp.pbody.push (h1);
      tmp.pbody.push (h2);
      tmp.print (out);
      tmp.clear ();
      tmp = *r;
      r->head.array[0].atom = h1;
      tmp.head.array[0].atom = h2;
      change_atmost_to_atleast (&tmp);
      tmp.print (out);
      tmp.clear ();
    }
}

int
Parser::weight (Rule *r)
{
  r->type = WEIGHTRULE;
  r->atleast = 0;
# ifdef USEDOUBLE
  r->atmost = DBL_MAX;
# else
  r->atmost = ULONG_MAX;
# endif
  for (;;)
    {
      int token = gettoken ();
      if (token == RULE_END)
	{
	  split_weight_rule (r);
	  return Ok;
	}
      if (!isWeight (gettoken ()))
	{
	  error ("I expected a number");
	  return Error;
	}
      Weight n = getWeight ();
      switch (token)
	{
	case EQUAL:
	  if (token == NEGNUMBER)
	    {
	      if (r->offset < n)
		r->atleast = 0;
	      else
		r->atleast = r->offset - n;
	    }
	  else
	    r->atleast = n+r->offset;
	  if (token == NEGNUMBER)
	    {
	      if (r->offset < n)
		r->type = NORULE;
	      r->atmost = r->offset - n;
	    }
	  else
	    r->atmost = n+r->offset;
	  break;
	case ATMOST:
	  if (token == NEGNUMBER)
	    {
	      if (r->offset < n)
		r->type = NORULE;
	      r->atmost = r->offset - n;
	    }
	  else
	    r->atmost = n+r->offset;
	  break;
	case ATLEAST:
	  if (token == NEGNUMBER)
	    {
	      if (r->offset < n)
		r->atleast = 0;
	      else
		r->atleast = r->offset - n;
	    }
	  else
	    r->atleast = n+r->offset;
	  break;
#       ifndef USEDOUBLE
	case LESS:
	  if (token == NEGNUMBER)
	    {
	      if (r->offset < n-1)
		r->type = NORULE;
	      r->atmost = r->offset - (n-1);
	    }
	  else
	    r->atmost = n-1+r->offset;
	  break;
	case GREATER:
	  if (token == NEGNUMBER)
	    {
	      if (r->offset <= n)
		r->atleast = 0;
	      else
		r->atleast = r->offset - (n+1);
	    }
	  else
	    r->atleast = n+1+r->offset;
	  break;
#       endif
	default:
	  error ("I expected a comparison operator");
	  return Error;
	}
    }
}

int
Parser::optimize (bool max)
{
  r.clear ();
  r.type = OPTIMIZERULE;
  if (braced (&r) == Error)
    return Error;
  if (max)
    r.swap_pos_neg ();
  r.print (out);
  return Ok;
}

int
Parser::braced (Rule *r)
{
  if (gettoken () != LBRACE)
    {
      error ("I expected a left brace");
      return Error;
    }
  if (body (r) == Error)
    return Error;
  if (gettoken () != RBRACE)
    {
      error ("I expected a right brace");
      return Error;
    }
  return Ok;
}

int
Parser::braced (Array *a, bool pos)
{
  if (gettoken () != LBRACE)
    {
      error ("I expected a left brace");
      return Error;
    }
  if (array (a, pos) == Error)
    return Error;
  if (gettoken () != RBRACE)
    {
      error ("I expected a right brace");
      return Error;
    }
  return Ok;
}

int
Parser::body (Rule *r)
{
  Atom *a;
  r->offset = 0;
  Weight w;
  bool positive;
  for (;;)
    {
      switch (gettoken ())
	{
	case DELIMITER:  // This accepts several delimiters in a row
	  break;
	case NEGATION:
	  a = atom ();
	  if (!a)
	    {
	      error ("The error is in the body of the rule");
	      return Error;
	    }
	  positive = true;
	  if (gettoken () == EQUAL)
	    {
	      int token = gettoken ();
	      if (!isWeight (token))
		{
		  error ("I expected a weight after the equal sign");
		  return Error;
		}
	      w = getWeight ();
	      if (token == NEGNUMBER)
		positive = false;
	    }
	  else
	    {
	      havetoken = true;
	      w = a->weight;
	      positive = a->positive_weight;
	    }
	  if (!positive)
	    {
	      r->offset += w;
	      r->pbody.push (a, w);
	    }
	  else
	    r->nbody.push (a, w);
	  break;
	case ATOM:
	  havetoken = true;
	  a = atom ();
	  if (!a)
	    {
	      error ("The error is in the body of the rule");
	      return Error;
	    }
	  positive = true;
	  if (gettoken () == EQUAL)
	    {
	      int token = gettoken ();
	      if (!isWeight (token))
		{
		  error ("I expected a weight after the equal sign");
		  return Error;
		}
	      w = getWeight ();
	      if (token == NEGNUMBER)
		positive = false;
	    }
	  else
	    {
	      havetoken = true;
	      w = a->weight;
	      positive = a->positive_weight;
	    }
	  if (!positive)
	    {
	      r->offset += w;
	      r->nbody.push (a, w);
	    }
	  else
	    r->pbody.push (a, w);
	  break;
	default:
	  havetoken = true;
	  return Ok;
	}
    }
}

int
Parser::array (Array *ar, bool pos)
{
  Atom *a;
  for (;;)
    {
      switch (gettoken ())
	{
	case DELIMITER:  // This accepts several delimiters in a row
	  break;
	case NEGATION:
	  if (pos)
	    {
	      error ("Negative literal not expected");
	      return Error;
	    }
	  a = atom ();
	  if (!a)
	    {
	      error ("Expected atom after not");
	      return Error;
	    }
	  ar->push (a);
	  break;
	case ATOM:
	  if (!pos)
	    {
	      error ("Positive literal not expected");
	      return Error;
	    }
	  havetoken = true;
	  a = atom ();
	  if (!a)
	    {
	      error ("Expected atom");
	      return Error;
	    }
	  ar->push (a);
	  break;
	default:
	  havetoken = true;
	  return Ok;
	}
    }
}

Atom *
Parser::newAtom (const char *s)
{
  Atom *a = new Atom (s);
  if (s)
    {
      Atom *b = tree.insert (a);
      if (b != a)
	{
	  delete a;
	  a = b;
	  return a;
	}
    }
  lastid++;
  a->id = lastid;
  atoms.push (a);
  return a;
}

Atom *
Parser::getFail ()
{
  if (fail == 0)
    fail = newAtom (0);
  return fail;
}

Atom *
Parser::atom ()
{
  atomname[0] = '\0';
  atomname[sizeof (atomname)-1] = '\0';
  if (getatom () == Error)
    return 0;
  else
    return newAtom (atomname);
}

int
Parser::getatom (bool parameter)
{
  int token = gettoken ();
  if ((parameter && !isParameter (token)) ||
      (!parameter && token != ATOM))
    {
      error ("I expected an atom");
      return Error;
    }
  strncat (atomname, yytext, sizeof (atomname)-1);
  if (gettoken () == LPAREN)
    {
      strncat (atomname, yytext, sizeof (atomname)-1);
      for (;;)
	{
	  if (getatom (true) == Error)
	    return Error;
	  switch (gettoken ())
	    {
	    case RPAREN:
	      strncat (atomname, yytext, sizeof (atomname)-1);
	      return Ok;
	    case DELIMITER:
	      strncat (atomname, yytext, sizeof (atomname)-1);
	      break;
	    default:
	      error ("I expected a right parenthesis or a delimiter");
	      return Error;
	    }
	}
    }
  else
    havetoken = true;
  return Ok;
}

int
Parser::compute ()
{
  long mm = 1;
  switch (gettoken ())
    {
    case ALL:
      mm = 0;
      break;
    case POSNUMBER:
      mm = getnumber ();
      if (mm < 0)
	{
	  error ("Number of models in compute statement must be positive");
	  return Error;
	}
      break;
    default:
      havetoken = true;
      break;
    }
  if (gettoken () == LBRACE)
    {
      havetoken = true;
      if (braced (&computeRule) == Error)
	return Error;
    }
  else
    havetoken = true;
  maxmodels = mm;
  return Ok;
}

void
Parser::print_line ()
{
}

void
Parser::print ()
{
  out << 0 << endl;
  for (Node *n = atoms.begin (); n; n = n->next)
    if (n->atom->name)
      out << n->atom->id << ' ' << n->atom->name << endl;
  out << 0 << endl;
  out << "B+" << endl;
  for (int i = 0; i < computeRule.pbody.top; i++)
    out << computeRule.pbody.array[i].atom->id << endl;
  out << 0 << endl;
  out << "B-" << endl;
  for (int i = 0; i < computeRule.nbody.top; i++)
    out << computeRule.nbody.array[i].atom->id << endl;
  out << 0 << endl;
  out << maxmodels << endl;
}
