// Copyright 1998 by Patrik Simons
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston,
// MA 02111-1307, USA.
//
// Patrik.Simons@hut.fi
#include <iostream.h>
#include <string.h>
#include <stdlib.h>
#include <float.h>
#include <limits.h>
#include "parser.h"
#include "tokens.h"

extern int yylex ();
extern char *yytext;
extern int yyleng;
extern long linenumber;
extern int computestatement;

Parser::Parser (ostream &o)
  : out (o)
{
  havetoken = false;
  token = 0;
  maxmodels = 1;
  lastid = 0;
  fail = 0;
}

Parser::~Parser ()
{
}

inline int
Parser::gettoken ()
{
  if (havetoken) {
    havetoken = false;
    return token;
  }
  token = yylex ();
  return token;
}

long
Parser::getnumber ()
{
  return strtol (yytext, 0, 0);
}

Weight
Parser::getWeight ()
{
# ifdef USEDOUBLE
  return strtod (yytext, 0);
# else
  return strtol (yytext, 0, 0);
# endif
}

int
Parser::program ()
{
  for (;;)
    {
      switch (gettoken ())
	{
	case COMPUTE:
	  if (compute () == Error)
	    return Error;
	  break;
	case MAXIMIZE:
	  if (optimize (true) == Error)
	    return Error;
	  break;
	case MINIMIZE:
	  if (optimize (false) == Error)
	    return Error;
	  break;
	case 0:
	  return Ok;  // End of file.
	default:
	  havetoken = true;
	  if (statement () == Error)
	    return Error;
	  break;
	}
    }
  return Ok;
}

void
Parser::error (char *s)
{
  cerr << s
       << " at line "
       << linenumber << endl;
  print_line ();
}

bool
Parser::isWeight (int t)
{
# ifdef USEDOUBLE
  return t == NUMBER || t == DOUBLE;
# else
  return t == NUMBER;
# endif
}

bool
Parser::isParameter (int t)
{
  return t == ATOM || t == NUMBER || t == PARAMETER;
}

int
Parser::statement ()
{
  int token = gettoken ();
  havetoken = true;
  if (token == NUMBER)
    return generate ();
  if (token == LBRACE)
    return choice ();
  Atom *a = atom ();
  if (!a)
    {
      error ("I expected an atom as head or weight");
      return Error;
    }
  switch (gettoken ())
    {
    case RULE_ARROW:
    case RULE_END:
      havetoken = true;
      return rule (a);
    case EQUAL:
      token = gettoken ();
      if (!isWeight (token))
	{
	  error ("I expected a weight");
	  return Error;
	}
      a->weight = getWeight ();
      break;
    }
  return Ok;
}

int
Parser::generate ()
{
  gettoken ();
  r.clear ();
  r.type = GENERATERULE;
  r.chooseHead = getnumber ();
  if (braced (&r.head, true) == Error)
    {
      error ("The error is in the head of the rule");
      return Error;
    }
  int h = r.head.size ();
  if (h < 2)
    {
      error ("I expected at least two atoms in the head");
      return Error;
    }
  if (r.chooseHead < 1 || r.chooseHead >= h)
    {
      error ("Choose in head out of bounds");
      return Error;
    }
  switch (gettoken ())
    {
    case RULE_ARROW:
      if (array (&r.pbody, true) == Error)
	return Error;
      if (gettoken () != RULE_END)
	{
	  error ("I expected a `.'");
	  return Error;
	}
      break;
    case RULE_END:
      break;
    default:
      error ("I expected a `:-' or a `.'");
      return Error;
    }
  r.print (out);
  return Ok;
}

int
Parser::choice ()
{
  r.clear ();
  r.type = CHOICERULE;
  if (braced (&r.head, true) == Error)
    {
      error ("The error is in the head of the rule");
      return Error;
    }
  int h = r.head.size ();
  if (h < 1)
    {
      error ("I expected at least one atoms in the head");
      return Error;
    }
  return rule (0);
}

int
Parser::rule (Atom *a)
{
  if (a)
    {
      r.clear ();
      r.type = BASICRULE;
      r.head.push (a);
    }
  bool ret = Ok;
  switch (gettoken ())
    {
    case RULE_ARROW:
      if (gettoken () == NUMBER)
	{
	  ret = constraint (&r);
	  break;
	}
      else
	havetoken = true;
      if (token == LBRACE)
	{
	  if (braced (&r) == Error)
	    return Error;
	  ret = weight (&r);
	  break;
	}
      if (body (&r) == Error)
	return Error;
      if (gettoken () != RULE_END)
	{
	  error ("I expected a `.'");
	  return Error;
	}
      break;
    case RULE_END:
      break;
    default:
      error ("I expected a `:-' or a `.'");
      return Error;
    }
  if (ret == Ok)
    r.print (out);
  return ret;
}

int
Parser::constraint (Rule *r)
{
  r->type = CONSTRAINTRULE;
  long n = getnumber ();
  if (n <= 0)
    {
      error ("I expected positive number on constraint");
      return Error;
    }
  if (braced (r) == Error)
    return Error;
  long ps = r->pbody.size ();
  long ns = r->nbody.size ();
  if (ps > 0 && ns > 0)
    {
      error ("I expected only one kind of literals in constraint");
      return Error;
    }
  if (ps)
    r->choosePos = n;
  else if (ns)
    r->chooseNeg = n;
  else
    {
      error ("I expected some literals in constraint");
      return Error;
    }
  if (gettoken () != RULE_END)
    {
      havetoken = true;
      if (gettoken () != NUMBER)
	{
	  error ("I expected a number in constraint");
	  return Error;
	}
      n = getnumber ();
      if (n <= 0)
	{
	  error ("I expected positive number in constraint");
	  return Error;
	}
      if (ps)
	{
	  if (braced (&r->nbody, false) == Error)
	    return Error;
	  r->chooseNeg = n;
	  n = r->nbody.size ();
	}
      else
	{
	  if (braced (&r->pbody, true) == Error)
	    return Error;
	  r->choosePos = n;
	  n = r->pbody.size ();
	}
      if (n == 0)
	{
	  error ("I expected some literals in constraint");
	  return Error;
	}
      if (gettoken () != RULE_END)
	{
	  error ("I expected a `.'");
	  return Error;
	}
    }
  return Ok;
}

void
Parser::remove_negative_weights (Rule *r)
{
# ifdef USEDOUBLE
  r->atmost = DBL_MAX;
# else
  r->atmost = LONG_MAX;
# endif
  Weight w;
  for (int i = 0; i < r->pbody.top;)
    {
      Atom *a = r->pbody.array[i].atom;
      if (r->pbody.array[i].hasWeight)
	w = r->pbody.array[i].weight;
      else
	w = r->pbody.array[i].atom->weight;
      if (w < 0)
	{
	  r->nbody.push (a, -w);
	  r->atleast += -w;
	  r->pbody.top--;
	  r->pbody.array[i] = r->pbody.array[r->pbody.top];
	}
      else
	i++;
    }
  for (int i = 0; i < r->nbody.top;)
    {
      Atom *a = r->nbody.array[i].atom;
      if (r->nbody.array[i].hasWeight)
	w = r->nbody.array[i].weight;
      else
	w = r->nbody.array[i].atom->weight;
      if (w < 0)
	{
	  r->pbody.push (a, -w);
	  r->atleast += -w;
	  r->nbody.top--;
	  r->nbody.array[i] = r->nbody.array[r->nbody.top];
	}
      else
	i++;
    }
}

void
Parser::change_atmost_to_atleast (Rule *r)
{
  r->atleast = -r->atmost;
# ifdef USEDOUBLE
  r->atmost = DBL_MAX;
# else
  r->atmost = LONG_MAX;
# endif
  for (int i = 0; i < r->pbody.top; i++)
    if (r->pbody.array[i].hasWeight)
      r->pbody.array[i].weight = -r->pbody.array[i].weight;
    else
      {
	r->pbody.array[i].weight = -r->pbody.array[i].atom->weight;
	r->pbody.array[i].hasWeight = true;
      }
  for (int i = 0; i < r->nbody.top; i++)
    if (r->nbody.array[i].hasWeight)
      r->nbody.array[i].weight = -r->nbody.array[i].weight;
    else
      {
	r->nbody.array[i].weight = -r->nbody.array[i].atom->weight;
	r->nbody.array[i].hasWeight = true;
      }
}

void
Parser::split_weight_rule (Rule *r)
{
# ifdef USEDOUBLE
  Weight w = -DBL_MAX;
# else
  Weight w = LONG_MIN;
# endif
  if (w == r->atleast)
    change_atmost_to_atleast (r);
# ifdef USEDOUBLE
  w = DBL_MAX;
# else
  w = LONG_MAX;
# endif
  if (w != r->atmost)
    {
      Atom *h1 = newAtom (0);
      Atom *h2 = newAtom (0);
      tmp.type = BASICRULE;
      tmp.head.push (r->head.array[0].atom);
      tmp.pbody.push (h1);
      tmp.pbody.push (h2);
      tmp.print (out);
      tmp.clear ();
      tmp = *r;
      r->head.array[0].atom = h1;
      remove_negative_weights (r);
      tmp.head.array[0].atom = h2;
      change_atmost_to_atleast (&tmp);
      remove_negative_weights (&tmp);
      tmp.print (out);
      tmp.clear ();
    }
  else
    remove_negative_weights (r);
}

int
Parser::weight (Rule *r)
{
  r->type = WEIGHTRULE;
# ifdef USEDOUBLE
  r->atleast = -DBL_MAX;
  r->atmost = DBL_MAX;
# else
  r->atleast = LONG_MIN;
  r->atmost = LONG_MAX;
# endif
  for (;;)
    {
      int token = gettoken ();
      if (token == RULE_END)
	{
	  split_weight_rule (r);
	  return Ok;
	}
      if (!isWeight (gettoken ()))
	{
	  error ("I expected a number");
	  return Error;
	}
      Weight n = getWeight ();
      switch (token)
	{
	case EQUAL:
	  r->atleast = n;
	  r->atmost = n;
	  break;
	case ATMOST:
	  r->atmost = n;
	  break;
	case ATLEAST:
	  r->atleast = n;
	  break;
#       ifndef USEDOUBLE
	case LESS:
	  r->atmost = n-1;
	  break;
	case GREATER:
	  r->atleast = n+1;
	  break;
#       endif
	default:
	  error ("I expected a comparison operator");
	  return Error;
	}
    }
}

int
Parser::optimize (bool max)
{
  r.clear ();
  r.type = OPTIMIZERULE;
  if (max)
    r.choosePos = 1;
  else
    r.choosePos = 0;
  if (braced (&r) == Error)
    return Error;
  r.print (out);
  return Ok;
}

int
Parser::braced (Rule *r)
{
  if (gettoken () != LBRACE)
    {
      error ("I expected a left brace");
      return Error;
    }
  if (body (r) == Error)
    return Error;
  if (gettoken () != RBRACE)
    {
      error ("I expected a right brace");
      return Error;
    }
  return Ok;
}

int
Parser::braced (Array *a, bool pos)
{
  if (gettoken () != LBRACE)
    {
      error ("I expected a left brace");
      return Error;
    }
  if (array (a, pos) == Error)
    return Error;
  if (gettoken () != RBRACE)
    {
      error ("I expected a right brace");
      return Error;
    }
  return Ok;
}

int
Parser::body (Rule *r)
{
  Atom *a;
  for (;;)
    {
      switch (gettoken ())
	{
	case DELIMITER:  // This accepts several delimiters in a row
	  break;
	case NEGATION:
	  a = atom ();
	  if (!a)
	    {
	      error ("The error is in the body of the rule");
	      return Error;
	    }
	  if (gettoken () == EQUAL)
	    {
	      int token = gettoken ();
	      if (!isWeight (token))
		{
		  error ("I expected a weight after the equal sign");
		  return Error;
		}
	      r->nbody.push (a, getWeight ());
	    }
	  else
	    {
	      havetoken = true;
	      r->nbody.push (a);
	    }
	  break;
	case ATOM:
	  havetoken = true;
	  a = atom ();
	  if (!a)
	    {
	      error ("The error is in the body of the rule");
	      return Error;
	    }
	  if (gettoken () == EQUAL)
	    {
	      int token = gettoken ();
	      if (!isWeight (token))
		{
		  error ("I expected a weight after the equal sign");
		  return Error;
		}
	      r->pbody.push (a, getWeight ());
	    }
	  else
	    {
	      havetoken = true;
	      r->pbody.push (a);
	    }
	  break;
	default:
	  havetoken = true;
	  return Ok;
	}
    }
}

int
Parser::array (Array *ar, bool pos)
{
  Atom *a;
  for (;;)
    {
      switch (gettoken ())
	{
	case DELIMITER:  // This accepts several delimiters in a row
	  break;
	case NEGATION:
	  if (pos)
	    {
	      error ("Negative literal not expected");
	      return Error;
	    }
	  a = atom ();
	  if (!a)
	    {
	      error ("Expected atom after not");
	      return Error;
	    }
	  ar->push (a);
	  break;
	case ATOM:
	  if (!pos)
	    {
	      error ("Positive literal not expected");
	      return Error;
	    }
	  havetoken = true;
	  a = atom ();
	  if (!a)
	    {
	      error ("Expected atom");
	      return Error;
	    }
	  ar->push (a);
	  break;
	default:
	  havetoken = true;
	  return Ok;
	}
    }
}

Atom *
Parser::newAtom (const char *s)
{
  Atom *a = new Atom (s);
  if (s)
    {
      Atom *b = tree.insert (a);
      if (b != a)
	{
	  delete a;
	  a = b;
	  return a;
	}
    }
  lastid++;
  a->id = lastid;
  atoms.push (a);
  return a;
}

Atom *
Parser::getFail ()
{
  if (fail == 0)
    fail = newAtom (0);
  return fail;
}

Atom *
Parser::atom ()
{
  atomname[0] = '\0';
  atomname[sizeof (atomname)-1] = '\0';
  if (getatom () == Error)
    return 0;
  else
    return newAtom (atomname);
}

int
Parser::getatom (bool parameter)
{
  int token = gettoken ();
  if ((parameter && !isParameter (token)) ||
      (!parameter && token != ATOM))
    {
      error ("I expected an atom");
      return Error;
    }
  strncat (atomname, yytext, sizeof (atomname)-1);
  if (gettoken () == LPAREN)
    {
      strncat (atomname, yytext, sizeof (atomname)-1);
      for (;;)
	{
	  if (getatom (true) == Error)
	    return Error;
	  switch (gettoken ())
	    {
	    case RPAREN:
	      strncat (atomname, yytext, sizeof (atomname)-1);
	      return Ok;
	    case DELIMITER:
	      strncat (atomname, yytext, sizeof (atomname)-1);
	      break;
	    default:
	      error ("I expected a right parenthesis or a delimiter");
	      return Error;
	    }
	}
    }
  else
    havetoken = true;
  return Ok;
}

int
Parser::compute ()
{
  long mm = 1;
  switch (gettoken ())
    {
    case ALL:
      mm = 0;
      break;
    case NUMBER:
      mm = getnumber ();
      if (mm < 0)
	{
	  error ("Number of models in compute statement must be positive");
	  return Error;
	}
      break;
    default:
      havetoken = true;
      break;
    }
  if (gettoken () == LBRACE)
    {
      havetoken = true;
      if (braced (&computeRule) == Error)
	return Error;
    }
  else
    havetoken = true;
  maxmodels = mm;
  return Ok;
}

void
Parser::print_line ()
{
}

void
Parser::print ()
{
  out << 0 << endl;
  for (Node *n = atoms.begin (); n; n = n->next)
    if (n->atom->name)
      out << n->atom->id << ' ' << n->atom->name << endl;
  out << 0 << endl;
  out << "B+" << endl;
  for (int i = 0; i < computeRule.pbody.top; i++)
    out << computeRule.pbody.array[i].atom->id << endl;
  out << 0 << endl;
  out << "B-" << endl;
  for (int i = 0; i < computeRule.nbody.top; i++)
    out << computeRule.nbody.array[i].atom->id << endl;
  out << 0 << endl;
  out << maxmodels << endl;
}
