// Copyright 1998 by Patrik Simons
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston,
// MA 02111-1307, USA.
//
// Patrik.Simons@hut.fi
#include <assert.h>
#include <string.h>
#include "tree.h"

Tree::Tree ()
{
  root = 0;
  size = 0;
}

Tree::~Tree ()
{
  flush (root);
}

void
Tree::flush (TreeNode *n)
{
  if (n->left)
    flush (n->left);
  if (n->right)
    flush (n->right);
  delete n;
}

Atom *
Tree::find(const char *key) const
{
  TreeNode *n = root;
  int r;
  while (n)
    {
      r = strcmp (key, n->atom->name);
      if (r < 0)
	n = n->left;
      else if (r > 0)
	n = n->right;
      else
	return n->atom;
    }
  if (n)
    return n->atom;
  else
    return 0;
}


//     pp(b)               p(b)
//    / \                 /  \
//   l   p(r)    ==>    pp(r) c(r)
//       / \           / \
//     pc   c(r)      l   pc
//
//  ppc points to p from pp
inline void
Tree::slope (TreeNode *p, TreeNode *&pc, TreeNode *&ppc)
{
  ppc = pc;
  pc->parent = p->parent;
  pc = p->parent;
  p->color = black;
  p->parent->color = red;
  if (p->parent == root)
    root = p;
  TreeNode *n = p->parent->parent;
  p->parent->parent = p;
  p->parent = n;
}

//        pp(b)                   c(b)
//          \                    /  \
//           p(r)             pp(r)  p(r)
//          /          ==>     \     /
//         c(r)               cppc cpc
//        / \
//     cppc cpc
//
//
inline void
Tree::zigzag (TreeNode *c, TreeNode *p, TreeNode *&pc, TreeNode *&cpc,
	      TreeNode *&ppc, TreeNode *&cppc)
{
  pc = cpc;
  cpc->parent = p;
  cpc = p;
  ppc = cppc;
  cppc->parent = p->parent;
  cppc = p->parent;
  p->parent->color = red;
  c->color = black;
  c->parent = p->parent->parent;
  if (p->parent == root)
    root = c;
  else
    {
      if (c->parent->left == p->parent)
	c->parent->left = c;
      else
	c->parent->right = c;
    }
  p->parent = c;
}

Atom *
Tree::insert (Atom *a)
{
  TreeNode *p = root;
  TreeNode *n = root;
  TreeNode *c = new TreeNode (a);
  c->color = red;
  int r;
  if (root == 0)
    {
      root = c;
      size++;
      a->id = size;
      return a;
    }
  while (n)
    {
      p = n;
      r = strcmp (a->name, n->atom->name);
      if (r < 0)
	n = n->left;
      else if (r > 0)
	n = n->right;
      else
	return n->atom;  // Already in tree.
    }
  size++;
  a->id = size;
  if (r < 0)
    p->left = c;
  else
    p->right = c;
  for (;;)
    {
      if (p->color == black)
	break;
      else if (p == root)
	{
	  root->color = black;
	  break;
	}
      assert (p->parent->color == black);  // Otherwise corrupt tree
      TreeNode *u;  // The uncle
      if (p->parent->left == p)
	u = p->parent->right;
      else
	u = p->parent->left;
      if (u && u->color == red)
	{
	  p->color = black;
	  p->parent->color = red;
	  u->color = black;
	  if (p->parent == root)
	    break;
	  // Up the tree
	  c = p->parent;
	  p = c->parent;
	}
      else
	{
	  if (p->left == c)  // Left child
	    if (p->parent->right == p)
	      zigzag (c, p, p->left, c->right, p->parent->right, c->left);
	    else
	      slope (p, p->right, p->parent->left);
	  else  // Right child
	    if (p->parent->left == p)
	      zigzag (c, p, p->right, c->left, p->parent->left, c->right);
	    else
	      slope (p, p->left, p->parent->right);
	  break;
	}
    }
  return a;
}
