// predicate.cc -- implementation of predicate data type
// Copyright (C) 1999-2000 Tommi Syrjnen <Tommi.Syrjanen@hut.fi>
//  
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//  
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//  
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//  

#include "../config.h"
#include <string.h>
#include <stdio.h>
#include <stdlib.h>


#ifndef PREDICATE_H
#include "predicate.h"
#endif
#ifndef SYMBOL_H
#include "symbol.h"
#endif
#ifndef ITERATE_H
#include "iterate.h"
#endif
#ifndef ERROR_H
#include "error.h"
#endif

Predicate::Predicate()
  : indices(NULL), atoms(NULL), existing_indexes(0), arity(-1),
    rules(UNORDERED), special_rules(UNORDERED),
    it(NULL), name(NULL), status(DM_UNKNOWN) 
{
  pred = -1;
  complement = -1;
  negation = -1;
  emitted = NULL;
  follows = 0;
  external_atom = 0;
  depends_on_external_atoms = 0;
  transitive_domain = 0;
  hidden = 0;
  shown = 0;
  special = 0;
  has_rule = 0;
  is_warned = 0;
  is_internal = 0;
  is_priority_predicate = 0;
  predicate_weight_list = NULL;
  weight_list = NULL;
  first_line = 0;
}


Predicate::~Predicate()
{
  int i;
  if (indices) {
    for (i = 0; i < arity; i++) {
      if (indices[i])
	delete indices[i];
    } 
    delete [] indices;
  }
  if (it)
    delete it;
  if (atoms)
    delete atoms;
  if (name && strcmp(name, FALSE_NAME))
    delete name;
}


// returns NULL if the predicate name is already complemented 
char *complement_name(char *nm)
{
  int len = strlen(nm);
  static char comp_name[BUFSIZ] = { 0 };

  if (len > BUFSIZ -2) {
    error(FATAL_ERR, "too long predicate name '%s'", nm);
  }

  if (nm[len-1] == '\'') // || nm[0] == '_' )
    return NULL;
  
  strcpy(comp_name, nm);
  strcpy(&comp_name[len], "'");
  return comp_name;
}

char *complement_atom_name(char *nm)
{
  static char comp_name[BUFSIZ] = { 0 };
  int pos = 0;
  
  while (*nm && *nm != '\'' && *nm != '(') {
    comp_name[pos++] = *nm++;
  }
  if (*nm == '\'')
    return NULL;
  
  comp_name[pos++] = '\'';
  while (*nm) {
    comp_name[pos++] = *nm++;
  }
  comp_name[pos++] = '\0';
  return comp_name;
}

  

// negate the atom name: foo -> -foo && -foo -> foo
char *negation_name(char *nm)
{
  int len = strlen(nm);
  static char neg_name[BUFSIZ] = { 0 };

  if (len > BUFSIZ -2) {
    error(FATAL_ERR, "too long predicate name '%s'", nm);
  }

  if (*nm == '-') {
    sprintf(neg_name, "%s", nm+1);
    neg_name[len-1] = '\0';
  } else {
    sprintf(neg_name, "-%s", nm);
    neg_name[len+1] = '\0';
  }
  return neg_name;
}

RestrictType Predicate::CheckRestrict(int print)
{
  debug(DBG_RESTRICT, 2, "Checking restrict for %s", name);
  assert(arity >= 0);
  RestrictType result = RT_STRONG, tmp = RT_STRONG;
  Rule *rl = NULL;

  if ((status == DM_SPECIAL) ||
      (status == DM_INTERNAL))
    return RT_STRONG;
  
  while ((rl = rules.Iterate())) {
    tmp = rl->CheckRestrict(print);
    switch (tmp) {
    case RT_NONE:
      result = RT_NONE;
      break;
    case RT_WEAK:
      if (result > RT_NONE)
	result = RT_WEAK;
      break;
    default:
      break;
    }
  }

  debug(DBG_RESTRICT, 2, "\tResult: $d", result);
  return result;
}

void Predicate::AddRule(Rule *rl, int)
{
  has_rule = 1;
  // check if the rule is a fact
  if ((rl->positive.Size() == 0) && (rl->negative.Size() == 0) &&
      (rl->functions.Size() == 0)) {
    AddInstance(rl->head->cons);

    //    if (sys_data.use_regular_models && cmp) {
    //      predicates[complement]->AddInstance(rl->head->cons);
    //    }
    
    delete rl;
  } else {
    rules.Insert(rl);
    //    if (sys_data.use_regular_models && cmp) {
    //      Rule *r = rl->CreateComplement();
    //      predicates[complement]->rules.Insert(r);
    //    }
  }
}


void Predicate::CreateComplementRules()
{
  Rule *rl = NULL;
  
  while ((rl = rules.Iterate())) {
    Rule *r = rl->CreateComplement();
    predicates[complement]->rules.Insert(r);
  }
  
  while ((rl = special_rules.Iterate())) {
    Rule *r = rl->CreateComplement();
    predicates[complement]->special_rules.Insert(r);
  }
}

void Predicate::CreateAllComplementRules()
{
  long p;
  for (p = 0 ; p < predicate_table->Size(); p++) {
    if (predicates[p]->complement > p) {
      predicates[p]->CreateComplementRules();
    }
  }
}

void Predicate::AddSpecialRule(Rule *rl, int )
{
  has_rule = 1;
  special = 1;

  special_rules.Insert(rl);

  // check if this is the internal false predicate that is otherwise a
  // domain predicate and false.
  if (false_lit && (false_lit->pred == pred) && (status == DM_DOMAIN)) {
    status = DM_FALSE;
  }

  //  if (sys_data.use_regular_models && cmp) {
  //    Rule *r = rl->CreateComplement();
  //    predicates[complement]->special_rules.Insert(r);
  //  }

}


void Predicate::AddInstance(Instance *inst)
{
  if (arity == 0)
    follows = 1;
  else 
    atoms->Insert(inst);
}

void Predicate::AddWeight(Weight *nw, Literal *lt, int global)
{
  WeightNode *wn = NULL;

  wn = new WeightNode;
  wn->wt = nw;
  wn->lt = lt;
  if (global) {
    wn->prev = predicate_weight_list;
    predicate_weight_list = wn;
  } else {
    wn->prev = weight_list;
    weight_list = wn;
  }
}


void Predicate::SetArity(int ar)
{
  arity = ar;

  if (ar > 0) {
    if (!atoms) {
      atoms = new InstanceSet(sys_data.domain_size, ar);
      it = new InstanceIterator(atoms);
      if (!atoms || ! it)
	error(SYS_ERR, "malloc error");
    } else {
      atoms->Clear(ar);
      it->Clear();
    }
  }
}



void Predicate::SetStatus(DomainType r)
{
  status = r;

}
  

void Predicate::CreateIndex(int ind)
{
  Instance *item = NULL;
  assert((ind >= 0) && (ind < arity));

  if (!indices) {
    indices = new Index*[arity];
    if (!indices)
      error(SYS_ERR, "malloc error");
    for (long i = 0; i < arity; i++ )
      indices[i] = NULL;
  }
  if (existing_indexes & (1 << ind)) 
    int_error("trying to create same index two times in predicate"
	      " '%s'", name);

  indices[ind] = new Index(arity, ind);
  if (!indices[ind])
    error(SYS_ERR, "malloc error");

  existing_indexes |= (1 << ind);

  it->Clear();
  while ((item = it->Iterate())) 
    indices[ind]->Insert(item);
  return;
}

char *get_phi_predicate(char *st, int positive)
{
  static char buf[BUFSIZ] = { 0 };
  size_t pos = 0;

  while (*st && (*st != '(')) {
    st++;
  }

  if (!st)
    int_error("misformed priority literal", "");

  pos += sprintf(buf, "phi_%s'", positive ? "pos" : "neg");

  strcpy (&buf[pos], st);

  return buf;
}



void Predicate::Emit()
{
  Instance *item = NULL;
  Rule *rl = NULL;
  Literal *head = NULL;
  static Instance *items = NULL;
  int i = 0;

  if (is_priority_predicate && !is_priority_tcl) {
    return;
  }
  
  // check if this is a domain predicate that shouldn't be printed
  if (!external_atom && !depends_on_external_atoms &&
      !is_priority_predicate &&
      (((sys_data.print_domains == PR_NONE) &&
	(status == DM_DOMAIN)) || (status == DM_INTERNAL) ||
       (status == DM_SPECIAL)))
    return;

  if (it)
    it->Clear();

  if (!items) {
    items = new Instance[predicate_table->MaxArity()+1];
    if (!items)
      error(SYS_ERR, "malloc error");
    memset(items, 0, sizeof(Instance) * predicate_table->MaxArity()+1); 
  }

  // don't print external atoms 
  if (!external_atom && !depends_on_external_atoms) {

    if (arity == 0) {
      if (follows) {
	// predicate is a ground fact. Construct a rule and emit it.
	head = new Literal(pred, NULL, 0, 0);
	if (!head)
	  error(SYS_ERR, "malloc error");
	rl = new Rule(head, BASICRULE);
	if (!rl)
	  error(SYS_ERR, "malloc error");
	rl->EmitGround(0,0,1);
	delete rl;
      } else if (false_lit && (false_lit->pred == pred) &&
		 (sys_data.output_version < 2)) {
	// construct the rule false :- false. (this is needed because 
	// otherwise smodels 1 won't work when false is not true :-)
	head = false_lit;
	rl = new Rule(head, BASICRULE);
	if (!rl)
	  error(SYS_ERR, "malloc error");
	rl->AddLiteral(head); 
	rl->EmitGround(0,0,0);
	delete rl;
      }
    }
    
    // print all the ground instances 
    if ((arity > 0) && atoms && (atoms->Size() > 0)) { 
      head = new Literal(pred, items, arity, 0);
      if (!head)
	error(SYS_ERR, "malloc error");
      
      rl = new Rule(head, BASICRULE);
      if (!rl)
	error(SYS_ERR, "malloc error");
      
      while ((item = it->Iterate())) {
	for (i = 0; i < arity; i++) {
	  head->cons[i] = item[i];
	}
	//	if (sys_data.use_priorities && is_priority_tcl) {
	//	  priority_atoms->Insert(head->GetInstanceString());
	//	}
	rl->EmitGround(0,0,1);
      }
      delete rl;
    }
  }
  rules.ClearIterator();
  // and the normal rules
  if (!external_atom &&
      (depends_on_external_atoms ||(status != DM_DOMAIN))) { 
    while ((rl = rules.Iterate())) {
      if ((sys_data.print_domains == PR_ALL) && rl->ground)
	rl->EmitGround(0,0,0);
      else
	rl->GroundRule(GRD_RULE);
    }
    while ((rl = special_rules.Iterate()))
      rl->GroundSpecialRule(GRD_RULE);
  }


  if (sys_data.use_priorities && is_priority_tcl) {
    
    char *st = 0;
    // create and output the priority rules
    if (!priority_rules) {
      char buf[BUFSIZ] = { 0 };
      size_t pos = 0;
      priority_rules = new StringList;

      for (i = 0; i < priority_atoms->Size(); i++) {
	st = priority_atoms->symbols[i];
	pos = 0;
	Instance l = GetArgumentInstanceFromString(st, 1);
	Instance r = GetArgumentInstanceFromString(st, 2);

	if ((l < 0) || (r < 0))
	  int_error("misformed priority atom","");


	if (sys_data.emit_text) {
	  pos += sprintf(buf, "%s :- %s, not %s, %s.\n",
			 get_phi_predicate(st, true),
			 st,
			 atom_table->symbols[r-1],
			 atom_table->symbols[l-1]);
	  pos += sprintf(&buf[pos], "%s :- %s,  %s, not %s.\n",
			 get_phi_predicate(st, false),
			 st,
			 atom_table->symbols[r-1],
			 atom_table->symbols[l-1]);

	  if (pos > BUFSIZ) {
	    fprintf(stderr, "buffer overflow\n");
	    abort();
	  }
	} else {
	  pos += sprintf(buf, "1 %ld 2 1 %ld %ld\n",
			 atom_table->Insert(get_phi_predicate(st,true))+1,
			 r,l);
	  pos += sprintf(&buf[pos], "1 %ld 2 1 %ld %ld\n",
			 atom_table->Insert(get_phi_predicate(st,false))+1,
			 l,r);
	}
	priority_rules->Insert(clone_string(buf));
      }

    }

    // print the priority rules
    while ((st = priority_rules->Iterate())) {
      fprintf(sys_data.output_file, "%s", st);
    }
  }
}

void Predicate::CalculateDomain()
{
  debug(DBG_GROUND, 2, "Calculating domain for %s", name);
  Rule *rl = NULL;
  int type = GRD_DOMAIN;
  
  // don't change status of internal range predicates
  if (status != DM_INTERNAL)
    SetStatus(DM_DOMAIN);
  else if (is_internal) {
    status = DM_INTERNAL;
  }
  
  if (transitive_domain)
    type = GRD_TRANSITIVE_DOMAIN;
  
  // ground all rules one at time
  rules.ClearIterator();
  while ((rl = rules.Iterate())) {
    rl->GroundRule(type);
  }

  if (type == GRD_TRANSITIVE_DOMAIN) {
    //    CreateTransitiveDomain();
  }
  
}
      

void Predicate::EmitAll()
{
  long i = 0;

  if (false_lit)
    atom_table->Insert(FALSE_NAME);

  for( i = 0; i < predicate_table->Size(); i++) {
    if (predicates[i]->external_atom)
      dependency_graph->PropagateExternalDefinitions(i);
  }
  
  for( i = 0; i < predicate_table->Size(); i++)
    predicates[i]->Emit();
}

void Predicate::PrintAllRules()
{
  long i = 0;
  Rule *rl = NULL;
  
  for (i = 0; i < predicate_table->Size(); i++) {
    while ((rl = predicates[i]->rules.Iterate())) {
      rl->PrintRule();
    }
  }
}


RestrictType Predicate::CheckAllRestricts()
{
  RestrictType rt = RT_STRONG, tmp = RT_STRONG;
  long i = 0;
  
  for( i = 0; i < predicate_table->Size(); i++){
    tmp = predicates[i]->CheckRestrict(1);
    switch (tmp) {
    case RT_NONE:
      rt = RT_NONE;
      break;
    case RT_WEAK:
      if (rt > RT_WEAK)
	rt = RT_WEAK;
      break;
    default:
      break;
    }
  }
  return rt;
}

void Predicate::DefineAllComplements()
{
  long i;
  DomainType tp;
  long orig_size = predicate_table->Size();
  
  if (false_lit) {
    predicates[false_lit->pred]->complement =
      false_lit->pred;
  }
  
  for (i = 0; i < orig_size; i++) {
    tp = predicates[i]->Status();
    if (tp == DM_FALSE) {
      DefineComplement(i);
    }
  }
  CreateAllComplementRules();
}


long Predicate::DefinePredicate(char *nm, int ar, long ln)
{
  long p = -1;

  // check whether the predicate is classically negated
  if (*nm == '-') {
    if (!sys_data.true_negation) {
      error(USR_ERR, "%ld: The minus sign in the predicate "
	   "`%s/%d' was interpreted as a classical negation but the "
	   "command line argument `--true-negation' was not used.\n"
	   "If you want it to be interpreted as an arithmetic minus "
	   "operation, please add a space between it and the "
	   "following symbol.", ln, nm, ar);
    }
  }

  p = predicate_table->Insert(nm, ar);

  
  if (p < 0)
    int_error("unknown predicate '%s'", nm);
  if (predicates[p]->pred == p)
    return p;  // already defined

  if (!strcmp(nm, FALSE_NAME)) {
    predicates[p]->SetStatus(DM_FALSE);
  } else if (sys_data.hide_all)
    predicates[p]->hidden = 1;
  
  
  dependency_graph->AddNode(p);
  predicates[p]->SetName(nm);
  predicates[p]->SetPred(p);
  predicates[p]->SetArity(ar);
  predicates[p]->first_line = ln;

  if (sys_data.use_priorities) {
    if ((ar == 2) && !strcmp(nm, PRIORITY_PREDICATE)) {
      predicates[p]->is_priority_predicate = 1;
    }
  }

  
#if 0
  if (sys_data.use_regular_models && !strchr(nm, '\'')) {
    if (!strcmp(nm, FALSE_NAME))
      predicates[p]->complement = p;
    else
      DefineComplement(p);
  }
#endif
  
  if (sys_data.true_negation) {
    DefineNegated(p);
    false_lit = Predicate::DefineFalseLiteral();
  }
  
  // check if there is a predicate with same name but different arity 
  if (sys_data.warnings & WARN_ARITY) {
    long p2 = predicate_table->CheckName(nm, ar);
    if (p2 >= 0) {
      warn(WARN_ARITY, predicates[p2]->first_line,
	   "predicate '%s' is used with %d argument%s at line %ld, while " 
	   "it is also used with %d argument%s at line %ld.",
	   nm, predicates[p2]->Arity(),
	   (predicates[p2]->Arity() == 1) ? "" : "s",
	   predicates[p2]->first_line,
	   ar,
	   (ar == 1) ? "" : "s",
	   ln);
      predicates[p2]->is_warned = 1;
    }
  }
  return p;
}

// generate a dummy head that cannot be true in any case. 
long Predicate::ConstructDummy()
{
  static char *buf = 0;
  int first = 0;
  
  if (!buf) {
    buf = clone_string("_dummy");
    first = 1;
  }
  long p = DefinePredicate(buf, 0, 1);
  if (first) {
    Literal *lt = new Literal();
    lt->pred = p;
    lt->negative = 1;
    compute->rl->AddLiteral(lt);
  }
  return p;
}

long Predicate::DefineSystemPredicate(DomainType tp)
{
  static char buf[RANGE_MAX_NAME] = { 0 }; 
  const int pos = 4;
  int val = 0;
  long p = -1;
  if (!buf[0]) {
    sprintf(buf, "_int");
  }
  
  val = sprintf(&buf[pos], "%ld", sys_data.internal_atoms++);
  if (val + pos > RANGE_MAX_NAME)
    int_error("too many special rules","");

  p =  DefinePredicate(clone_string(buf), 0, 0);
  predicates[p]->SetStatus(tp);
  predicates[p]->is_internal =1;
  
  if (sys_data.true_negation) {
    predicates[predicates[p]->negation]->SetStatus(tp);
    predicates[predicates[p]->negation]->is_internal = 1;
  }

  
  //  if (sys_data.use_regular_models) {
  //    DefineComplement(p);
  //  }
  dependency_graph->AddNode(p);
  return p;
}

long Predicate::DefinePriorityPredicate()
{
  long user_priority = predicate_table->Lookup(PRIORITY_PREDICATE, 2);

  long help_predicate = DefineRangePredicate();
  predicates[help_predicate]->SetStatus(DM_UNKNOWN);
  predicates[help_predicate]->is_priority_predicate = 1;
  if (user_priority < 0) {
    int_error("missing priority predicate definition","");
  }
  
  long p = DefinePredicate(clone_string(PRIORITY_TCL), 2,
			   predicates[user_priority]->first_line);

  predicates[p]->is_priority_predicate = 1;
  predicates[p]->is_priority_tcl = 1;
  
  Variable v[2] = { 0 }, v1, v2, v3;
  v1 = variable_table->Insert("I_1");
  v2 = variable_table->Insert("I_2");
  v3 = variable_table->Insert("I_3");

  // help_predicate(I_1) :- prefer(I_1, I_2).
  v[0] = v1;
  v[1] = v2;

  Literal *hd = new Literal(v[0], help_predicate);
  hd->lineno = predicates[p]->first_line;

  Rule *rl = new Rule(hd, BASICRULE);

  Literal *lt = new Literal(v, user_priority, 2);
  lt->lineno = predicates[p]->first_line;
  rl->AddLiteral(lt);
  
  predicates[help_predicate]->AddRule(rl);

  hd = new Literal(v[1], help_predicate);
  hd->lineno = predicates[p]->first_line;

  rl = new Rule(hd, BASICRULE);

  lt = new Literal(v, user_priority, 2);
  lt->lineno = predicates[p]->first_line;
  rl->AddLiteral(lt);
  
  predicates[help_predicate]->AddRule(rl);

  // priority'(I_1, I_3) :- priority'(I_1, I_2), priority(I_2, I_3).
  v[0] = v1;
  v[1] = v3;

  hd = new Literal(v, p, 2);
  hd->lineno = predicates[p]->first_line;

  
  rl = new Rule(hd, BASICRULE);

  v[0] = v1;
  v[1] = v2;
  
  lt = new Literal(v, p, 2);
  lt->lineno = predicates[p]->first_line;
  rl->AddLiteral(lt);

  v[0] = v2;
  v[1] = v3;
  
  lt = new Literal(v, user_priority, 2);
  lt->lineno = predicates[p]->first_line;
  rl->AddLiteral(lt);

  lt = new Literal(v1, help_predicate);
  lt->lineno = predicates[p]->first_line;
  rl->AddLiteral(lt);

  Term *l = new Term(T_VARIABLE, v1, predicates[p]->first_line);
  Term *r = new Term(T_VARIABLE, v2, predicates[p]->first_line);
  Function *fun = new Function("neq", l, r,
			       predicates[p]->first_line);
  rl->AddFunction(fun);
  l = new Term(T_VARIABLE, v1, predicates[p]->first_line);
  r = new Term(T_VARIABLE, v3, predicates[p]->first_line);
  fun = new Function("neq", l, r,
			       predicates[p]->first_line); 
  rl->AddFunction(fun);
  
  predicates[p]->AddRule(rl);

  // priority'(I_1, I_2) :- priority(I_1, I_2);
  v[0] = v1;
  v[1] = v3;
  
  hd = new Literal(v, p, 2);
  rl = new Rule(hd, BASICRULE);

  lt = new Literal(v, user_priority, 2);
  lt->lineno = predicates[p]->first_line;
  rl->AddLiteral(lt);

  
  predicates[p]->AddRule(rl);

  dependency_graph->AddEdge(p, user_priority);
  dependency_graph->AddEdge(p, p);
  dependency_graph->AddEdge(p, help_predicate);
  return p;
}


long Predicate::DefineRangePredicate()
{
  static char buf[RANGE_MAX_NAME] = { 0 }; 
  const int pos = 4;
  int val = 0;
  long p = -1;
  if (!buf[0]) {
    sprintf(buf, "_int");
  }
   
  val = sprintf(&buf[pos], "%ld", sys_data.internal_atoms++);
  if (val + pos > RANGE_MAX_NAME)
    int_error("too many special rules","");
 
  p = DefinePredicate(clone_string(buf), 1, 0);
  predicates[p]->SetStatus(DM_INTERNAL);
  predicates[p]->is_internal = 1;

  //  if (sys_data.use_regular_models) {
  //    predicates[predicates[p]->complement]->is_internal = 1;
  //    predicates[predicates[p]->complement]->SetStatus(DM_INTERNAL);
  //  }
  return p;
}


long Predicate::DefineInconsistentPredicate()
{
  static char buf[RANGE_MAX_NAME] = { 0 }; 
  long p = -1;
  if (!buf[0]) {
    sprintf(buf, "%s", INCONSISTENT_NAME);
  }
  
  p =  DefinePredicate(clone_string(buf), 0, 0);
  predicates[p]->SetStatus(DM_FALSE);
  predicates[p]->is_internal =1;
  
  return p;
}


Literal *Predicate::DefineFalseLiteral()
{
  long p = DefinePredicate(FALSE_NAME, 0, 0);
  Literal *new_lt = new Literal(p, NULL, 0,0);
  atom_table->Insert(FALSE_NAME);
  predicates[p]->SetStatus(DM_FALSE);
  predicates[p]->is_internal = 1;
  return new_lt;
}

// initialize a complementary predicate to allow regular model
// semantics to work 
void Predicate::DefineComplement(long p)
{
  // sanity check
  if (predicates[p]->pred != p)
    int_error("trying to define a complement of unexisting predicate","");

  long comp = -1;
  Predicate *original = predicates[p];

  // already defined?
  if (original->complement >= 0)
    return;

  char *new_name = complement_name(original->Name());
  if (!new_name) // complemented already 
    return; 
  comp = DefinePredicate(clone_string(new_name), original->Arity(), original->first_line);
  
  original->complement = comp;
  predicates[comp]->complement = p;

  if (original->Internal()) {
    predicates[comp]->is_internal = 1;
  }
  
  dependency_graph->AddEdge(comp, p);
  dependency_graph->AddEdge(p, comp);
  
  return;
}

void Predicate::DefineNegated(long p)
{
  // sanity check
  if (predicates[p]->pred != p)
    int_error("trying to define a negation of unexisting predicate","");


  long neg = -1;
  Predicate *original = predicates[p];

  // already defined?
  if (original->negation >= 0)
    return;

  char *new_name = negation_name(original->Name());

  neg = DefinePredicate(clone_string(new_name), original->Arity(),
			original->first_line); 

  original->negation = neg;
  predicates[neg]->negation = p;

  if (original->Internal()) {
    predicates[neg]->is_internal = 1;
  }
}



long Predicate::Size()
{
  if (!atoms)
    return 0;
  else
    return atoms->Size();
}

void Predicate::EmitComplements()
{
  long at = 0;
  long cmpl = 0;
  char *at_st = NULL;
  char *complement_st = NULL;

  for (at = 0; at < atom_table->Size(); at++) {
    at_st = atom_table->symbols[at];
    complement_st = complement_atom_name(at_st);

    if (complement_st) {
      if (sys_data.emit_text) {
	fprintf(sys_data.output_file, "%s :- %s.\n", complement_st, at_st);
	if (sys_data.regular_level == REGULAR_ALL_CONSTRAINTS) {
	  fprintf(sys_data.output_file, "%s :- %s.\n", at_st ,
		  complement_st);
	}
      } else {
	cmpl = atom_table->Insert(complement_st);
	//	if (cmpl < 0) {
	//	  int_error("missing complementement", "");
	//	}
	
	if (sys_data.output_version >= 2) {
	  fprintf(sys_data.output_file, "%d %ld 1 0 %ld\n",
		  BASICRULE, cmpl+1, at+1);
	  if (sys_data.regular_level == REGULAR_ALL_CONSTRAINTS) {
	    fprintf(sys_data.output_file, "%d %ld 1 0 %ld\n",
		    BASICRULE, at+1, cmpl+1);
	  }
	} else {
	  fprintf(sys_data.output_file, "%ld  1 %ld 0\n",
		  cmpl+1, at +1);
	  if (sys_data.regular_level == REGULAR_ALL_CONSTRAINTS) {
	    fprintf(sys_data.output_file, "%ld  1 %ld 0\n",
		    at +1, cmpl+1);
	  }
	}
      }
    }
  }
}

void Predicate::EmitNegations()
{
  long at = 0;
  long neg = 0;
  long false_at = atom_table->Lookup(FALSE_NAME) +1;
  long inconsistent_at = -1;
  char *at_st = NULL;
  char *neg_st = NULL;

  if (sys_data.inconsistent_answers)
    inconsistent_at = atom_table->Insert(INCONSISTENT_NAME) + 1;
  
  for (at = 0; at < atom_table->Size(); at++) {
    at_st = atom_table->symbols[at];

    if (*at_st == '-') {
      neg_st = atom_table->symbols[at] +1;

      if (sys_data.emit_text) {

	fprintf(sys_data.output_file, "%s :- %s, %s.\n",
		sys_data.inconsistent_answers ? INCONSISTENT_NAME : FALSE_NAME ,
		neg_st, at_st);
      } else {
	neg = atom_table->Insert(neg_st);
	if (neg < 0) {
	  int_error("missing complementement", "");
	}
	
	if (sys_data.output_version >= 2) {
	  fprintf(sys_data.output_file, "%d %ld 2 0 %ld %ld\n",
		  BASICRULE, sys_data.inconsistent_answers ?
		  inconsistent_at : false_at,  neg+1, at+1);

	} else {
	  fprintf(sys_data.output_file, "%ld 2 %ld %ld 0\n",
		  sys_data.inconsistent_answers ? inconsistent_at :
		  false_at, neg+1, at+1);
	 
	}
      }
    } else if (sys_data.totalize_negations) {
      long p = decode_atom(at_st);
      if (!predicates[p]->is_internal) {
	neg_st = negation_name(at_st);
	
	if (sys_data.emit_text) {
	  fprintf(sys_data.output_file, "%s :- not %s.\n",
		  neg_st, at_st);
	} else {
	  neg = atom_table->Insert(neg_st);
	  if (neg < 0) {
	    int_error("missing complementement", "");
	  }
	  
	  if (sys_data.output_version >= 2) {
	    fprintf(sys_data.output_file, "%d %ld 1 1 %ld\n",
		    BASICRULE, neg+1, at+1);
	  } else {
	    fprintf(sys_data.output_file, "%ld 1 %ld 0\n",
		    neg+1, at+1);
	    
	  }
	}
      }
    }
  }
}

void Predicate::CheckUnsatisfiable()
{
  for (long i = 0; i < predicate_table->Size(); i++) {
    if (!predicates[i]->Internal() && !predicates[i]->has_rule) {
      warn(WARN_UNSAT, predicates[i]->first_line,
	   "predicate '%s/%d' doesn't occur in any rule head.", 
	   predicates[i]->Name(), predicates[i]->Arity());
    }
  }
}

int Predicate::NeedsToBePrinted()
{
  return external_atom || depends_on_external_atoms;
}



long Predicate::decode_atom(char *st)
{
  long pos = 0;
  int paren_depth = 0;
  int inside_quotes = 0;
  int args = 0;

  char name_string[BUFSIZ];
  char *p;

  // find out name
  for (p = st; *p && (*p != '(') ; p++) {
    name_string[pos++] = *p;
  }
  name_string[pos] = '\0';

  // find out arity
  if (*p) {
    args++;
    p++;
    for ( ; *p; p++) {
      if ( (*p == ',') && !inside_quotes && (paren_depth == 0)) {
	args++;
      } else if (*p == '\"') {
	inside_quotes = !inside_quotes;
      } else if (*p == '(' && !inside_quotes) {
	paren_depth++;
      } else if (*p == ')' && !inside_quotes) {
	paren_depth--;
      }
    }
  }
  return DefinePredicate(clone_string(name_string), args, -1);
}

// returns the ground instance that corresponds to the n:th argument
// of the predicate string st or -1 if none exists
Instance Predicate::GetArgumentInstanceFromString(char *st, int n)
{
  Instance result = -1;
  char buf[BUFSIZ] = { 0 };
  char *p = st;
  size_t pos = 0;
  int paren_depth = 0;
  int current_argument = 1;
  int end = 0, end_inner = 0;
  int inside_quotes = 0;
  
  // search the beginning of the first argument;
  while (*p && (*p != '(')) {
    p++;
  }

  if (!*p) {
    return -1;
  }
  p++;
  // find the beginning of the correct argument
  while(!end && (current_argument < n)) {
    end_inner = 0;
    
    while (!end_inner && *p) {
      switch (*p) {
      case '(':
	if (!inside_quotes)
	  paren_depth++;
	break;
      case ')':
	if (!inside_quotes) {
	  paren_depth--;
	  if (paren_depth < 0) {
	    end = 1;
	  }
	}
	break;
      case ',':
	if (!inside_quotes) {
	  if (paren_depth == 0) {
	    current_argument++;
	    end_inner = 1;
	  }
	}
	break;
      case '"':
	inside_quotes = !inside_quotes;
	break;
      default:
	break;
      }
      p++;
    }
  }

  if (current_argument < n) { // too few arguments
    return -1;
  }

  st = p;
  // find the end of the argument
  pos = 0;
  end = 0;

  while (!end && *p) {
    switch (*p) {
    case '(':
      if (!inside_quotes) {
	paren_depth++;
      }
      break;
    case ')':
      if (!inside_quotes) {
	paren_depth--;
	if (paren_depth < 0) {
	  end = 1;
	}
      }
      break;
    case ',':
      if (!inside_quotes) {
	end = 1;
      }
      break;
    case '"':
      inside_quotes = !inside_quotes;
      break;
    default:
      break;
    }
    pos++;
    p++;
  }
  pos--;
  strncpy(buf, st, pos);
  buf[pos] = '\0';
  
  result = atom_table->Insert(buf);
  return result +1;
}

int Predicate::Internal()
{
  return is_internal;
}
