/*
   parse.y -- the bison parser for lparse

   This program has no warranties of any kind. Use at own risk.

   Author: Tommi Syrjnen (tommi.syrjanen@hut.fi)

   $Id: parse.y,v 1.1 1998/08/04 09:18:12 tssyrjan Exp $	 
*/

%{

#include "parse.h"
#include "debug.h"
#include "term.h"
#include "symbol.h"
#include "global.h"
#include <stdlib.h>
#include <string.h>
#include <stdio.h>

extern int yylex(void);
extern ParseNode *logic_program;

// Turn YYDEBUG on while debugging
#ifdef DEBUG
#define YYDEBUG 1
#else
#define YYDEBUG 0
#endif

#define YYERROR_VERBOSE

long lineno = 1;
int parse_errors = 0;
int return_symbol = 0;
int return_funcsymbol = 1;
void yyerror(char *st)
{
  parse_errors++;
  fprintf(stderr, "%ld: %s\n", lineno, st);
  exit(EXIT_FAILURE);
}

%}

%union {
   char *s;
   long l;
   ParseNode *pn;
}

%token <l> NUMBER
%token <s> FUNCTION, FUNCSYMBOL, IF, NOT, COMPUTE, IDENTIFIER,
%token <s> VARIABLE, DEFINE, SYMBOL, CONSTANT
%token <s> LE, GE, EQ, NEQ, DOTS, LT, GT, ASSIGN, MOD
%type <pn> program, declarations, declaration, rules, rule, head,
%type <pn> tail, literal, atom, function, arglist, term, compute_stmt,
%type <pn> models, c_list, literal_list, range, expression, constant,
/* %type <pn> deflist , defhead */

%left ASSIGN
%left EQ, LE, GE, LT, GT, NEQ
%left '+', '-'
%left '*', '/', MOD
%right UMINUS

%%

program : declarations rules
	{logic_program = new ParseNode(PT_PROGRAM, NULL, NULL, $2, lineno);}
	;

declarations : declarations declaration
	{ $$ = NULL; }
	| { $$ = NULL;}
	;

declaration : FUNCTION {return_funcsymbol = 0;} IDENTIFIER '.'
	{ function_table->Insert($3); return_funcsymbol = 1;}
	| DEFINE { return_symbol = 1; } SYMBOL SYMBOL  '.'
	{ function_table->Define($3, $4); return_symbol = 0; }
	;


rules :
	rules rule
	{ $$ = new ParseNode(PT_RULES, NULL,  $2, $1, lineno); }
	| rules compute_stmt
	{ $$ = new ParseNode(PT_RULES, NULL,  $2, $1, lineno); }
	| 
	{ $$ = NULL; }
	;

rule :
	head '.'
	{ $$ = new ParseNode(PT_RULE, NULL, $1, NULL, lineno); }
	| head IF tail '.'
	{ $$ = new ParseNode(PT_RULE, NULL, $1, $3, lineno); }
	| IF tail '.'
	{ $$ = new ParseNode(PT_RULE, NULL, NULL,$2, lineno); }
	;

/* defhead : head
	{ $$ = $1; }
	| IDENTIFIER '(' '[' deflist ']' ')'
	{ $$ = new ParseNode(PT_ATOM, $1, NULL, deflist, lineno); } 
	;

deflist : constant
	{ $$ = new ParseNode(PT_DEFLIST, NULL, $1, NULL, lineno); }
	| deflist constant
	{ $$ = new ParseNode(PT_DEFLIST, NULL, $2, $1, lineno); }
	;
*/
head : atom
	{ $$ = $1; }
	;

tail :
	tail ',' literal
	{ $$ = new ParseNode(PT_TAIL, NULL, $3, $1, lineno); }
	| literal
	{ $$ = new ParseNode(PT_TAIL, NULL, $1, NULL, lineno); } 
	| tail ',' expression
	{ $$ = new ParseNode(PT_TAIL, NULL, $3, $1, lineno); }
	| expression 
	{ $$ = new ParseNode(PT_TAIL, NULL, $1, NULL, lineno); } 
	;

literal :
	atom
	{ $$ = $1; }
	| NOT atom
	{ $$ = $2; $$->type = PT_NEG_ATOM; } 
	;

atom :	
	IDENTIFIER
	{ $$ = new ParseNode(PT_ATOM, $1, NULL, NULL, lineno); }
	| IDENTIFIER '(' arglist ')'
	{ $$ = new ParseNode(PT_ATOM, $1, NULL, $3, lineno); }
	;

function :
	FUNCSYMBOL '(' arglist ')'
	{ $$ = new ParseNode(PT_FUNCTION, $1, NULL, $3, lineno);}
	| NOT FUNCSYMBOL '(' arglist ')'
	{ $$ = new ParseNode(PT_FUNCTION, $2, NULL, $4, lineno, 1);}
	;

arglist :
	arglist ',' term
	{ $$ = new ParseNode(PT_ARGLIST, NULL, $3, $1, lineno); }
	| term
	{ $$ = new ParseNode(PT_ARGLIST, NULL, $1, NULL , lineno); }
	;

term :
	 constant
	{ $$ = $1; }
	| range
	{ $$ = $1;}
	| expression
	{ $$ = $1; }
	;

constant :
	 IDENTIFIER
	{ $$ = new ParseNode(PT_CONSTANT, $1, NULL, NULL, lineno);}
	;


range : NUMBER DOTS NUMBER
	{ $$ = new ParseNode(PT_RANGE, NULL, NULL, NULL, lineno, $1,
			     $3); } 
	;

expression :
	'(' expression ')'
	{ $$ = $2; }
	| '|' expression '|'
	{ $$ = new ParseNode(PT_EXPR, NULL, NULL, $2, lineno, FUN_ABS);}
	| '-' expression %prec UMINUS
	{ $$ = new ParseNode(PT_EXPR, NULL, NULL, $2 , lineno, FUN_MINUS);}
	| expression EQ expression
	{ $$ = new ParseNode(PT_EXPR, NULL, $1, $3, lineno, FUN_EQ);}
	| expression LE expression
	{ $$ = new ParseNode(PT_EXPR, NULL, $1, $3, lineno, FUN_LE);}
	| expression GE expression
	{ $$ = new ParseNode(PT_EXPR, NULL, $1, $3, lineno, FUN_GE);}
	| expression LT expression
	{ $$ = new ParseNode(PT_EXPR, NULL, $1, $3, lineno, FUN_LT);}
	| expression GT expression
	{ $$ = new ParseNode(PT_EXPR, NULL,  $1, $3, lineno, FUN_GT);}
	| expression NEQ expression
	{ $$ = new ParseNode(PT_EXPR, NULL,  $1, $3, lineno, FUN_NEQ);}
	| expression ASSIGN expression
	{ $$ = new ParseNode(PT_EXPR, NULL,  $1, $3, lineno, FUN_ASSIGN);}
	| expression '+' expression
	{ $$ = new ParseNode(PT_EXPR, NULL,  $1, $3, lineno, FUN_PLUS);}
	| expression '-' expression
	{ $$ = new ParseNode(PT_EXPR, NULL,  $1, $3, lineno, FUN_MINUS);}
	| expression '*'  expression
	{ $$ = new ParseNode(PT_EXPR, NULL,  $1, $3, lineno, FUN_TIMES);}
	| expression '/' expression
	{ $$ = new ParseNode(PT_EXPR, NULL, $1, $3, lineno, FUN_DIV);}
	| expression MOD expression
	{ $$ = new ParseNode(PT_EXPR, NULL,  $1, $3, lineno, FUN_MOD);}
	| VARIABLE
	{ $$ = new ParseNode(PT_VARIABLE, $1, NULL, NULL, lineno); }
	| NUMBER
	{ $$ = new ParseNode(PT_NUMBER, NULL, NULL, NULL, lineno,
			     $1);}
	| CONSTANT
	{ $$ = new ParseNode(PT_CONSTANT, $1, NULL, NULL, lineno);}
	| function
	{ $$ = $1; }
	;

models :
	NUMBER
	{ $$ = new ParseNode(PT_MODELS, NULL, NULL, NULL, lineno, $1); }
	| IDENTIFIER
	{ $$ = new ParseNode(PT_MODELS, $1, NULL, NULL, lineno, -1);}
	|
	{ $$ = NULL; }
	;


compute_stmt : COMPUTE models '{' c_list '}'
	{ $$ = new ParseNode(PT_COMPUTE_STMT, NULL, $2, $4, lineno);} 
	;


c_list :
	literal_list
	{ $$ = $1; }
	|
	{ $$ = NULL;}
	;

literal_list :
	literal_list ',' literal
	{ $$ = new ParseNode(PT_LITERAL_LIST, NULL, $3, $1, lineno);}
	| literal
	{ $$ = new ParseNode(PT_LITERAL, NULL, $1, NULL, lineno); }
	;
