/*
  parse.cc -- implementation of parse tree structure for lparse

  This program has no warranties of any kind. Use at own risk.
  
  Author: Tommi Syrjnen (tommi.syrjanen@hut.fi)
  
  $Id: parse.cc,v 1.1 1998/08/04 09:19:09 tssyrjan Exp tssyrjan $	 
  */

#include "parse.h"
#include "global.h"
#include "predicate.h"
#include "rule.h"
#include "debug.h"
#include "graph.h"
#include "symbol.h"
#include <stdio.h>
#include <stdlib.h>
#include <string.h>



//define strings that correspond to node types char
char *parse_strings[] = { "program", "declarations", "declaration",
			  "rules", "rule", "head", "tail", "literal",
			  "atom", "function", "arglist", "term",
			  "compute_stmt", "models", "clist",
			  "literal_list", "range", "neg_atom",
			  "constant", "expr", "variable", "number",
			  "not_atom", "deflist", "unknown" }; 
#ifdef DEBUG
// for tree printing
inline void indent(int amount)
{
  while (amount-- > 0)
    fprintf(stderr, " ");
}
#endif    
    
ParseNode::ParseNode(ParseType pt, char *v, ParseNode *l, ParseNode
		     *r, long ln)
  : type(pt), lval(-1), start(-1), end(-1), left(l), right(r),
    lineno(ln) 
{
  if (v)
    sval = strdup(v);
  else
    sval = NULL;
}

ParseNode::ParseNode(ParseType pt, char *v, ParseNode *l, ParseNode
		     *r, long ln, long lv)
  : type(pt), lval(lv), start(-1), end(-1), left(l), right(r),
    lineno(ln) 
{
  if (v)
    sval = strdup(v);
  else
    sval = NULL;
}

ParseNode::ParseNode(ParseType pt, char *v, ParseNode *l, ParseNode
		     *r, long ln, long s, long e)
  : type(pt), lval(-1), start(s), end(e), left(l), right(r),
    lineno(ln) 
{
  if (v)
    sval = strdup(v);
  else
    sval = NULL;
  ext.fun = NULL;
}
	
ParseNode::ParseNode()
  : type(PT_UNKNOWN), sval(NULL), lval(-1), start(-1), end(-1),
    left(NULL), right(NULL), lineno(-1) 
{
  ext.fun = NULL;
}
     
// Post-order walkthrough
int ParseNode::ProcessTree()
{
  int lresult = 0, rresult = 0;
  if (left)
    lresult = left->ProcessTree();

  if (right)
    rresult = right->ProcessTree();

  switch (type) {
    // these don't need any special treatment
  case PT_RANGE:
    return 1;
  case PT_CLIST:
  case PT_LITERAL:
  case PT_LITERAL_LIST:
  case PT_PROGRAM:
  case PT_RULES:
  case PT_ARGLIST:
  case PT_TERM:
  case PT_MODELS:
  case PT_HEAD:
  case PT_TAIL:
  case PT_DEFLIST:
    break;
    
  case PT_RULE:
    return ProcessRule(lresult, rresult);
    
  case PT_COMPUTE_STMT:
    c_stmt = this;
    break;
    //    return ProcessCompute(lresult, rresult);
    
  case PT_NEG_ATOM:
  case PT_ATOM:
  case PT_NOT_ATOM:
    return ProcessAtom(lresult, rresult);
    
  case PT_FUNCTION:
    return ProcessFunction(lresult, rresult);

  case PT_CONSTANT:
  case PT_VARIABLE:
  case PT_NUMBER:
  case PT_EXPR:
    return ProcessTerm(lresult, rresult);
  default:
    int_error("misformed parse tree: node type '%s'", parse_strings[type]);
  }
  return lresult + rresult;
}
  
  

int ParseNode::ProcessTerm(int , int )
{
  Term *lt = NULL, *rt = NULL;
  InstFunc fp = NULL;
  static Term *zero = NULL;
  char *op = NULL;
  
  switch (type) {
  case PT_CONSTANT:
    constant_table->Insert(sval);
    break;
  case PT_VARIABLE:
    variable_table->Insert(sval);
    break;
  case PT_NUMBER:
    // check if the number is biggest seen this far
    if (sys_data.max_number < lval)
      sys_data.max_number = lval;
    break;
  case PT_EXPR:
    // construct the call tree of the function and store it to 'fun'.

    // first get arguments
    if (left)
      lt = Term::ConstructArgument(left);
    else if (((InternalFunction) lval != FUN_MINUS) &&
	     ((InternalFunction) lval != FUN_ABS))
      int_error("operator '%d' without left operand", lval);
    else { // if uminus set left arg to zero
      if (!zero) {
	zero = new Term(T_CONSTANT, MAKE_NUMBER(0), lineno);
	if (!zero)
	  error(SYS_ERR, "malloc error");
      }
      lt = zero;
    }
    // check that nobody tries to assign to something that is not a
    // variable
    if (((InternalFunction)lval == FUN_ASSIGN) &&
	(lt->type != T_VARIABLE))
      error(USR_ERR, "%ld: invalid lvalue in assignment", lineno); 
    
    if (right)
      rt = Term::ConstructArgument(right);
    else 
      int_error("operator '%d' without right operand", lval);
    
    // get operator
    op = function_strings[lval];
    if (!op)
      int_error("Unknown operator '%d'", lval);
    fp = function_table->GetFunction(op);
    if (!fp)
      int_error("unknown function '%s'", op);
    ext.fun = new Function(fp, op, lineno);
    if ((InternalFunction)lval != FUN_ABS)
      ext.fun->AddArgument(lt);

    ext.fun->AddArgument(rt);
    
    if (!ext.fun)
      error(SYS_ERR, "malloc error");
    break;

  default:
    int_error("misformed parse tree:"
	      "Invalid term %ld", parse_strings[type]);
  }
  return 1;
}

int ParseNode::ProcessFunction(int , int )
{
  Term *arg = NULL;
  ParseNode *tmp = NULL, *parg = NULL;
  InstFunc fp = NULL;
  int arity = 0;
  
  // Find the pointer to function
  fp = function_table->GetFunction(sval);

  if (!fp)
    int_error("unknown function %s", sval);

  // allocate the function
  ext.fun = new Function(fp, sval, lineno);
  if (!ext.fun)
    error(SYS_ERR, "malloc error");

  // process the arguments
  tmp = right;
  while (tmp && (arity++ < TERM_MAX_ARITY)) {
    parg = tmp->left;
    if (!parg)
      int_error("Internal error: misformed parse tree."
		"missing function argument", "");
    switch(parg->type) {
    case PT_CONSTANT:
      if (!strcmp(sval, "eq") && !strcmp(sval, "neq"))
	warn(WARN_ARG, "%ld: function '%s' has a non numeric"
	     "constant '%s' argument", parg->lineno, sval,
	     parg->sval); 
    case PT_FUNCTION:
    case PT_EXPR:
    case PT_VARIABLE:
    case PT_NUMBER:
    case PT_RANGE:
      arg = Term::ConstructArgument(parg);
      break;

    default:
      int_error("misformed parse tree: "
	    "invalid function argument type %s",
	    parse_strings[parg->type]); 
    }
    ext.fun->AddArgument(arg);
    tmp = tmp->right;
  }
  if (arity >= TERM_MAX_ARITY)
    error(USR_ERR, "%ld: Maximum term arity %d exceeded", lineno,
	  TERM_MAX_ARITY);

  if (lval > 0)
    ext.fun->negative = 1;
  return 1;
}  

int ParseNode::ProcessCompute(int , int )
{
  assert(type == PT_COMPUTE_STMT);
  ParseNode *tmp = NULL;
  char *buf = NULL;
  long at = 0;
    
  // get number of models
  if (!left)
    compute->models = 0;
  else if (left->lval >= 0)
    compute->models = GET_VALUE(left->lval);
  else if (!strcmp(left->sval, "all"))
    error(USR_ERR, "%ld: Invalid number of models \"%s\"\n", lineno,
	  left->sval); 
  
  // check if all literals are ground literals and add them to
  // compute lists
  tmp = right;
  while (tmp) {
    if (!tmp->left)
      int_error("literal missing from compute statement", "");
    if (!(tmp->left->ext.lit->ground)) // weed out nonground literals
      error(USR_ERR, "%ld: Non ground literal '%s' in compute"
	    "statement\n", tmp->left->ext.lit->GetPrintString());
    else {
      buf = tmp->left->ext.lit->GetPrintString();
      at = atom_table->Insert(buf);
      if (at < 0)
	int_error("cannot generate compute literal '%s'", buf);
      
      // add atom to the compute lists
      if (tmp->left->type == PT_ATOM)
	compute->pos.Insert(at+1);
      else
	compute->neg.Insert(at+1);
    }
    tmp = tmp->right;
  }
  return 1;
}

int ParseNode::ProcessAtom(int , int rresult)
{
  long pred = -1;
  int arity = rresult; // rresult has the length of argument list
  char *name = NULL;

  switch (type) {
  case PT_NEG_ATOM:
    // for predicate ~foo() construct matching foo'()
    //    name = new char[strlen(sval)+2];
    // sprintf(name, "%s'", sval);
    //name[strlen(sval)+1] = '\0';
    /* FALL THROUGH */
  case PT_ATOM:
    // insert the symbol to the predicate table ...
    if (!name)
      name = sval;
    pred = predicate_table->Insert(name, arity);
    // ... and to dependency graph
    if (pred < 0)
      int_error("unknown predicate '%s'", name);
    if (predicates[pred]->Pred() != pred) {
      dependency_graph->AddNode(pred);
      predicates[pred]->SetName(name);
      predicates[pred]->SetPred(pred);
      predicates[pred]->SetArity(arity);
      lval = rresult;

      // save information about complement predicates foo and foo'. 
      //      if (type == PT_NEG_ATOM) {
      //	Predicate::MakeComplement(pred, sval, arity);
      // }
    }
    // and finally construct the literal
    ext.lit = new Literal(this, pred, arity);
    if (!ext.lit)
      error(SYS_ERR, "malloc error");
    return 1;
  case PT_NOT_ATOM:
    /* this doesn't work so it's commented out
       // transfer the actual literal "one level up" in parse tree
       ext.lit = left->ext.lit;
       ext.lit->negative = 1;
       type = left->type;
       return 1; */
  default:
    int_error("misformed parse tree", "");
  }
  return 1;
}

Literal *false_lit = NULL;

int ParseNode::ProcessRule(int , int )
{
  Rule *new_rule = NULL;
  long pred = -1, start = lineno, end = -1;
  Literal *head = NULL;
  ParseNode *tmp;
  
  if (!left && !false_lit) {
    pred = predicate_table->Insert(FALSE_NAME, 0);
    if (pred < 0)
      int_error("Cannot insert default false in symbol table", ""); 
    false_lit = new Literal;
    if (!false_lit)
	error(SYS_ERR, "malloc error");
    false_lit->ground = 1;
    false_lit->pred = pred;
    dependency_graph->AddNode(pred);
    predicates[pred]->SetName(FALSE_NAME);
    predicates[pred]->SetArity(0);
    predicates[pred]->SetPred(pred);
    // and insert it to atom table and compute lists also
    pred = atom_table->Insert(FALSE_NAME);
    if (pred < 0)
      int_error("Cannot insert default false in atom table", "");
    compute->neg.Insert(pred + 1);
  }
  
  // first save head, if in use
  if (left)
    head = left->ext.lit;
  else {
    // use default head
    head = false_lit;
  }
  if (pred < 0) 
    pred = head->pred;
    
  // Initialize the rule
  new_rule = new Rule(head);

  // go through the tail
  tmp = right;
  while (tmp) {
    if (!tmp->left)
      int_error("missing literal from a rule", "");
    if (tmp->left->lineno > end)
      end = tmp->left->lineno;
    switch (tmp->left->type) {
    case PT_ATOM:
    case PT_NEG_ATOM:
      new_rule->AddLiteral( tmp->left->ext.lit,
			    tmp->left->ext.lit->negative);
      dependency_graph->AddEdge(head->pred, tmp->left->ext.lit->pred);
      break;
    case PT_FUNCTION:
    case PT_EXPR:
      new_rule->AddFunction( tmp->left->ext.fun );
      break;
    default:
      int_error("misformed parse tree", "");
    }
    tmp = tmp->right;
  }
  new_rule->line_start = start;
  new_rule->line_end = end;
  predicates[head->pred]->AddRule(new_rule);
  return 1;
}
  
  
#ifdef DEBUG
// recursively print the tree in preorder
void ParseNode::Print(int ind)
{
  indent(ind);
  fprintf(stderr, "Type: %s\n", parse_strings[type]);
  if (sval) {
    indent(ind);
    fprintf(stderr, "SVal: %s\n", sval);
  }
  if (lval >= 0) {
    indent(ind);
    if (type != PT_EXPR)
      fprintf(stderr, "LVal: %ld\n", lval);
    else
      fprintf(stderr, "LVal: %s\n", function_strings[lval]);
  }

  if (start >= 0) {
    indent(ind);
    fprintf(stderr, "Range: %ld .. %ld\n", start, end);
  }
  indent(ind);
  fprintf(stderr, "Lineno: %ld\n", lineno);

  if (left){
    indent(ind);
    fprintf(stderr, "left:\n");
    left->Print(ind +3);
  }
  if (right) {
    indent(ind);
    fprintf(stderr, "right:\n");
    right->Print(ind+3);
  }
}

#endif


