/*
  main.cc -- for lparse
  
  This program has no warranties of any kind. Use at own risk.
  
  Author: Tommi Syrjnen (tommi.syrjanen@hut.fi)
  
  $Id: main.cc,v 0.9 1998/08/04 09:19:13 tssyrjan Exp tssyrjan $	 
*/

#include "global.h"
#include "symbol.h"
#include "library.h"
#include "parse.h"
#include "graph.h"
#include "predicate.h"

#include <stdio.h>
#include <ctype.h>
#include <stdlib.h>
#include <string.h>
#ifdef UNIX
#include <unistd.h>
#endif

// returns the position of first input file in command line
extern int yyparse();
int process_arguments(int argc, char *argv[]);

// lnitialize_static initializes data that doesn't depend on the size
// of the program to parse, initialize_dynamic initializes the rest. 
int initialize_dynamic();
int initialize_static();
void delete_vars();

// print out the output stored in a temporary file
void print_output();

char version[] = "lparse version $Id: main.cc,v 0.9 1998/08/04 09:19:13 tssyrjan Exp tssyrjan $";

int main(int argc, char *argv[])
{
  RestrictType res = RT_STRONG;
  initialize_static();
  process_arguments(argc, argv);
  register_functions();
  yyparse();

  if (sys_data.num_errors > 0) {
    fprintf(stderr, "%d errors found. exiting ... ", sys_data.num_errors);
    exit(E_ERROR);
  }
  
  if (!logic_program) {
    warn(WARN_ALL, "Empty program");
    exit(E_ERROR);
  }
  logic_program->ProcessTree();
  initialize_dynamic();

  dependency_graph->CheckCycles();

  res = Predicate::CheckAllRestricts();

  switch (res) {
  case RT_NONE:
    exit(E_ERROR);
  case RT_WEAK:
    exit(E_RANGE);
  default:
    break;
  }

  Predicate::EmitAll();

  if (!sys_data.emit_text)
    print_output();

  delete_vars();
  
  exit(E_OK);
}

int process_arguments(int argc, char *argv[])
{
  int i = 0, neg = 0;
  FILE *tmp_file = NULL, *read_file = NULL;
  char *tmp = NULL, *tmp_name = NULL;
  char buf[BUFSIZ] = { 0 } ;
  
  char *usage_msg = "usage: %s [-d all|heads|none] [-s size][-text]"
    "[-w warning] [input files]\n"
    "\t-d -- which domain predicates should be emitted:\n"
    "\t\tall - all domain predicates are emitted\n"
    "\t\tfacts - domain predicates in rule tails are omitted\n"
    "\t\tpositive - negative domain predicates are omitted\n"
    "\t\tnone - all domain predicates are omitted\n"
    "\t-s -- expected size of domains\n"
    "\t-text -- output plain text\n"
    "\t-f -- print out all functions that are registered and exit\n"
    "\t-i -- disable internal functions \n"
    "\t-v -- print version information\n"
    "\t-w warning -- enable/disable warnings\n"
    "\t\t[no]all - turn on/off all warnings\n"
    "\t\t[no]declarations - problems with function declarations\n"
    "\t\t[no]constant-args - warn of nonnumerical arguments\n"
    "\t\t[no]size - warn of invalid domain sizes\n"
    "\t\terror - treat warnings as errors\n"
    "\nThe default commandline is %s -d heads -s 100 -w all\n"
    "\nThe input file with function declarations should be listed"
    " first in the \ncommand line.\n";


  for (i=1; i < argc; i++) {
    tmp = argv[i];
    if (*tmp != '-')  // start of input files
      break;
    tmp++;
    switch(*tmp) {
    case 'f':
    case 'F':
      function_table->PrintRegistered();
      exit(0);
    case 'v':
      printf("%s\n", version);
      exit(0);
    case 'd':
    case 'D':
      i++;
      tmp = argv[i];
      if (!tmp)
	error(FATAL_ERR, usage_msg, argv[0]);
      switch(*tmp) {
      case 'a':
      case 'A':
	sys_data.print_domains = PR_ALL;
	break;
      case 'f':
      case 'F':
	sys_data.print_domains = PR_HEADS;
	break;
      case 'n':
      case 'N':
	sys_data.print_domains = PR_NONE;
	break;
      case 'p':
      case 'P':
	sys_data.print_domains = PR_POSITIVE;
	break;
      default:
	error(FATAL_ERR, usage_msg, argv[0]);
      }
      break;
    case 'i':
    case 'I':
      sys_data.internal_functions = 0;
      break;
    case 's':
    case 'S':
      i++;
      tmp = argv[i];
      if (!tmp || !isdigit(*tmp))
	error(FATAL_ERR, usage_msg, argv[0]);
      sys_data.domain_size = strtol(tmp, NULL, 10);
      if (sys_data.domain_size < DOMAIN_MIN_SIZE) {
	warn(WARN_SIZE, "Initial domain size too small (%ld)\n" 
	     "Increasing it to %d", sys_data.domain_size,
	     DOMAIN_MIN_SIZE ); 
	sys_data.domain_size = 20;
      } else if(sys_data.domain_size > DOMAIN_MAX_SIZE) {
	warn(WARN_SIZE, "Initial domain size too big (%ld)"
	     "Lowering it to %ld", sys_data.domain_size,
	     DOMAIN_MIN_SIZE ); 
	sys_data.domain_size = DOMAIN_MAX_SIZE;
      }
      break;
    case 't':
    case 'T':
      sys_data.emit_text = 1;
      break;
    case 'w':
    case 'W':
      i++;
      tmp = argv[i];
      if (!*tmp)
	error(FATAL_ERR, usage_msg, argv[0]);
      if (*tmp == 'n') {
	neg = 1;
	tmp += 2;
      }
      switch (*tmp) {
      case 'a':
	if (neg)
	  sys_data.warnings ^= WARN_ALL;
	else
	  sys_data.warnings |= WARN_ALL;
	break;
      case 'd':
	if (neg)
	  sys_data.warnings ^= WARN_DECL;
	else
	  sys_data.warnings |= WARN_DECL;
	break;
      case 'c':
	if (neg)
	  sys_data.warnings ^= WARN_ARG;
	else
	  sys_data.warnings |= WARN_ARG;
	break;
      case 's':
	if (neg)
	  sys_data.warnings ^= WARN_SIZE;
	else
	  sys_data.warnings |= WARN_SIZE;
	break;
      case 'e':
	sys_data.abort_when_warn = 1;
	break;
      default:
	error(FATAL_ERR, usage_msg, argv[0]);
      }
      neg = 0;
      break;
    default:
      error(FATAL_ERR, usage_msg, argv[0]);
    }
  }
  if (i == argc)
    return 1;
  
  // treat rest command arguments as filenames, write them to a
  // temporary file, rewind it and bind stdin to the new file
  tmp_name = tmpnam(NULL);
  if (!tmp_name)
    error(SYS_ERR, "tmp file creation error");

  tmp_file = fopen(tmp_name, "w");

  if (!tmp_file)
    error(SYS_ERR, "cannot open tmp file");
  
  for (; i < argc; i++) {
    read_file = fopen(argv[i], "r");
    if (!read_file)
      error(SYS_ERR, "cannot open file %s", argv[i]);
    while (fgets(buf, BUFSIZ, read_file))
      fprintf(tmp_file, "%s",  buf);
    fclose(read_file);
  }

  fclose(tmp_file);

  freopen(tmp_name, "r", stdin);
#ifdef UNIX
  unlink(tmp_name);
#endif
  return 1;
}
  
int initialize_static()
{
  debug(DBG_MAIN, 1, "initialize_static");
  sys_data.num_errors = 0;
  sys_data.emit_text = 0;
  sys_data.abort_when_warn = 0;
  sys_data.warnings = WARN_ALL;
  sys_data.internal_functions = 1;
  sys_data.print_domains = PR_HEADS;
  sys_data.domain_size = DOMAIN_DEFAULT_SIZE;
  sys_data.max_number = 0;
  sys_data.ground_atoms = 0;
  sys_data.ground_rules = 0;
  sys_data.ground_sigma = 0;
  
  function_table = new FunctionTable;
  predicate_table = new SymbolTable;
  constant_table = new SymbolTable;
  variable_table = new SymbolTable;
  atom_table = new SymbolTable;
  dependency_graph = new Graph;
  compute = new ComputeStmt;
  compute->models = 0;
  
  if (!function_table || !predicate_table || !constant_table || !
      atom_table || !variable_table || ! compute)
    error(SYS_ERR, "malloc error");



#ifdef DEBUG
  if (DEBUG & DBG_PARSE) {
    extern int yydebug;
    yydebug = 1;
  } 
#endif
  return 1;
}

int initialize_dynamic()
{
  debug(DBG_MAIN, 1, "initialize_dynamic");
  predicate_table->CreateSymbolArray();
  variable_table->CreateSymbolArray();
  constant_table->CreateSymbolArray();

  if (c_stmt)
    c_stmt->ProcessCompute(0,0);
  
  variables = new Instance[variable_table->Size()];
  if (!variables)
    error(SYS_ERR, "malloc error");

  if (sys_data.emit_text) {
    sys_data.output_file = stdout;
  } else {
    sys_data.output_file = tmpfile();
    if (!sys_data.output_file)
      error(SYS_ERR, "cannot create temporary file");
  }
  return 1;
}

void print_output()
{
  long at, i;
  char buf[BUFSIZ] = { 0 };
  // print the size information
  printf("%ld\n", sys_data.ground_atoms);
  printf("%ld\n", sys_data.ground_rules);
  printf("%ld\n", sys_data.ground_sigma);
  
  // reset tmpfile
  rewind(sys_data.output_file);

  // print out the rules
  while(fgets(buf, BUFSIZ, sys_data.output_file) != NULL)
    printf(buf);

  fclose(sys_data.output_file);
  // create symbols array for grounded atoms
  atom_table->CreateSymbolArray();

  // print out the atoms
  for (i = 0; i < atom_table->Size(); i++) {
    printf("%s\n", atom_table->symbols[i]);
  }

  // print compute stmt
  printf("B+\n");
  printf("%ld\n", compute->pos.Size());

  while ((at = compute->pos.Iterate())) {
    printf("%ld\n", at);
  }

  printf("B-\n");
  printf("%ld\n", compute->neg.Size());

  while ((at = compute->neg.Iterate())) {
    printf("%ld\n", at);
  }
  // and number of models
  printf("%ld\n", compute->models);
}

void delete_vars()
{
  delete function_table;
  delete predicate_table;
  delete constant_table;
  delete variable_table;
  delete atom_table;
  delete dependency_graph;
  delete compute;
  delete [] variables;
}
  

