/* list.cc -- implementation of lparse's list classes

   This program has no warranties of any kind. Use at own risk.

   Author: Tommi Syrjnen (tommi.syrjanen@hut.fi)

   $Id: list.cc,v 1.1 1998/08/04 09:19:19 tssyrjan Exp $	 
   */
   
#include "list.h"
#include <stdlib.h>
#include "debug.h"
#include <limits.h>

List::List(ListType tp)
{
  type = tp;
  last = items = CreateEmptyNode();
  current = NULL;
  if (!items)
    error(SYS_ERR, "malloc error");
  size = 0;
  last_weigth = last->weight;
}

List::~List()
{
  ListNode *tmp;

  while (items) {
    tmp = items;
    items = items->next;
    delete tmp;
  }
}

ListNode *List::CreateEmptyNode()
{
  ListNode *new_node = NULL;

  new_node = new ListNode;

  if (!new_node)
    error(SYS_ERR, "malloc error");

  new_node->item = NULL;
  new_node->next = NULL;
  if (type == ORDERED_BIG)
    new_node->weight = LONG_MAX;
  else
    new_node->weight = LONG_MIN;
  return new_node;
}

ListNode *List::DuplicateNode(ListNode *nd)
{
  assert(nd != NULL);
  ListNode *new_node = NULL;
  new_node = new ListNode;

  if (! new_node)
    error(SYS_ERR, "malloc error");

  new_node->item = nd->item;
  new_node->weight = nd->weight;
  new_node->next = NULL;
  return new_node;
}

void List::Insert(void *item, long weight)
{
  ListNode *new_node = NULL;
  ListNode *tmp = items;

  new_node = new ListNode;
  if (! new_node)
    error(SYS_ERR, "malloc error");

  new_node->weight = weight;
  new_node->item = item;
  new_node->next = NULL;
  size++;

  debug(DBG_LIST, 3, "List::Insert(), weight: %d, size: %d",
	weight, size);
  
  switch (type) {
  case UNORDERED:
    new_node->next = items->next;
    items->next = new_node;
    break;
  case ORDERED_BIG:
    if (weight < last->weight) {
      last->next = new_node;
      last = new_node;
      return;
    }
    while (tmp->next && (tmp->next->weight > weight))
      tmp = tmp->next;
    new_node->next = tmp->next;
    tmp->next = new_node;
    break;
  case ORDERED_SMALL:
    if (weight > last->weight) {
      last->next = new_node;
      last = new_node;
      current = items->next;
      return;
    }
    while (tmp->next && (tmp->next->weight < weight))
      tmp = tmp->next;
    new_node->next = tmp->next;
    tmp->next = new_node;
    break;
  }
  current = items->next;
  if (new_node->next == NULL) {
    last = new_node;
    last_weigth = new_node->weight;
  }
  return;
};

// Ei pida jarjestysta kunnossa
void List::Append(void *item, long wght)
{
  ListNode *new_node = NULL;
  new_node = new ListNode;

  if (!new_node)
    error(SYS_ERR, "malloc error");
  new_node->weight = wght;
  new_node->item = item;
  new_node->next = NULL;

  last->next = new_node;
  last = new_node;
  last_weigth = wght;
  current = items->next;
  size++;
}

void *List::Iterate()
{
  ListNode *tmp = current;

  if (!current) {
    current = items->next;
    return NULL;
  }
  current = current->next;
  return tmp->item;
}

void List::Merge(List *lst)
{
  debug(DBG_LIST, 3, "List::Merge");

  if (type == UNORDERED)
    MergeUnordered(lst);
  else
    MergeOrdered(lst);
}

void List::MergeUnordered(List *lst)
{
  debug(DBG_LIST, 3, "List::MergeUnordered");
  ListNode *tmp1 = last, *tmp2 = NULL;

  tmp2 = lst->items;
  size += lst->size;
  
  while (tmp2->next) {
    tmp1->next = DuplicateNode(tmp2->next);
    tmp1 = tmp1->next;
    tmp2 = tmp2->next;
  }
  current = items->next;
}

// Should x be before y in list
#define BEFORE(x, y) ((type == ORDERED_SMALL) ? \
		      ((x) < (y)) : ((x) > (y))) 

void List::MergeOrdered(List *lst)
{
  debug(DBG_LIST, 3, "List::MergeOrdered()");
  ListNode *new_list = NULL, *tmpn = NULL, *tmp1 = NULL, *tmp2 = NULL;
  
  new_list = CreateEmptyNode();
  if (! new_list)
    error(SYS_ERR, "malloc error");
  tmpn = new_list;

  tmp1 = items->next;
  tmp2 = lst->items->next;

  // Size is initialized to total lengths of both lists and then
  // subrtacted by one for each duplicate.
  size += lst->size;
  
  for (;;) {
    if (!tmp1) {  
      debug(DBG_LIST, 3, "List 1 finished");
      while (tmp2) {
	tmpn->next = DuplicateNode(tmp2);
	tmp2 = tmp2->next;
	tmpn = tmpn->next;
	last = tmpn;
      }
      break;
    } else if (!tmp2){  
      debug(DBG_LIST, 3, "List 2 finished");
      while (tmp2) {
	tmpn->next = DuplicateNode(tmp2);
	tmp2 = tmp2->next;
	tmpn = tmpn->next;
	last = tmpn;
      }
      break;
    } else if (BEFORE(tmp1->weight, tmp2->weight)) {
      debug(DBG_LIST, 3, "\t %d / %d chose 1", tmp1->weight, tmp2->weight);
      tmpn->next = DuplicateNode(tmp1);
      tmpn = tmpn->next;
      tmp1 = tmp1->next;
      last = tmpn;
    } else if (BEFORE(tmp2->weight, tmp1->weight)) {
      debug(DBG_LIST, 3, "\t %d / %d chose 2", tmp1->weight, tmp2->weight);
      tmpn->next = DuplicateNode(tmp2);
      tmpn = tmpn->next;
      tmp2 = tmp2->next;
      last = tmpn;
    } else {
      debug(DBG_LIST, 3, "\t %d / %d ...", tmp1->weight, tmp2->weight);
      if (tmp1->item == tmp2->item) {  // is duplicate?
	debug(DBG_LIST, 3, "\t equal");
	size--;
	tmp2 = tmp2->next;
      }
      tmpn->next = DuplicateNode(tmp1);
      tmpn = tmpn->next;
      tmp1 = tmp1->next;
      last = tmpn;
    }
  }

  // Delete the old list
  tmp1 = items;
  while (tmp1) {
    tmp2 = tmp1;
    tmp1 = tmp1->next;
    delete tmp2;
  }

  items = new_list;
  current = items->next;
  last_weigth = last->weight;
}
    
void List::Clear()
{
  ListNode *tmp = NULL;

  current= items->next;
  
  while(current) {
    tmp = current->next;
    delete current;
    current = tmp;
  }

  size = 0;
  last_weigth = 0;
  items->next = NULL;
  last = items;
}

	  
RuleList::RuleList(ListType tp)
{
  type = tp;
  last = current = items = CreateEmptyNode();
  last_weigth = last->weight;
  if (!items)
    error(SYS_ERR, "malloc error");
  size = 0; 
}



RuleList::~RuleList()
{
  ListNode *tmp = NULL;

  while (items) {
    tmp = items;
    items = items->next;
    delete tmp;
  }
}


void RuleList::Insert(Rule *item, long weight)
{
  List::Insert((void *) item, weight);
}

void RuleList::Append(Rule *item, long weight)
{
  List::Append((void *) item, weight);
}

Rule *RuleList::Iterate()
{
  return (Rule *) List::Iterate();
}


LiteralList::LiteralList(ListType tp)
{
  type = tp;
  last = current = items = CreateEmptyNode();
  if (!items)
    error(SYS_ERR, "malloc error");
  last_weigth = last->weight;
  size = 0; 
}


LiteralList::~LiteralList()
{
  ListNode *tmp;

  while (items) {
    tmp = items;
    items = items->next;
    delete tmp;
  }
}


void LiteralList::Insert(Literal *item, long weight)
{
  List::Insert((void *) item, weight);
}


Literal *LiteralList::Iterate()
{
  return (Literal*) List::Iterate();
}



LongList::LongList(ListType tp)
{
  type = tp;
  last = current = items = CreateEmptyNode();
  if (!items)
    error(SYS_ERR, "malloc error");
  last_weigth = last->weight;
  size = 0; 
}


LongList::~LongList()
{
  ListNode *tmp;

  while (items) {
    tmp = items;
    items = items->next;
    delete tmp;
  }
}


void LongList::Insert(long item, long weight)
{
  List::Insert((void *) item, weight);
}


long LongList::Iterate()
{
  return (long) List::Iterate();
}


FunctionList::FunctionList(ListType tp)
{
  type = tp;
  last = current = items = CreateEmptyNode();
  if (!items)
    error(SYS_ERR, "malloc error");
  last_weigth = last->weight;
  size = 0; 
}


FunctionList::~FunctionList()
{
  ListNode *tmp;

  while (items) {
    tmp = items;
    items = items->next;
    delete tmp;
  }
}


void FunctionList::Insert(Function *item, long weight)
{
  List::Insert((void *) item, weight);
}


Function *FunctionList::Iterate()
{
  return (Function*) List::Iterate();
}

