/* instance.cc -- Implementation of instances
   
   This program has no warranties of any kind. Use at own risk.

   Author: Tommi Syrjnen (tommi.syrjanen@hut.fi)

   $Id: instance.cc,v 1.1 1998/08/04 09:19:34 tssyrjan Exp $	 
*/

#include "instance.h"
#include "debug.h"
#include <string.h>
#include "global.h"
#include "symbol.h"

char *get_ground_instance(Instance *it, long pred, int ar)
{
  static char buf[BUFSIZ] = { 0 };
  int pos = 0;
  int i;
  pos += sprintf(&buf[pos], "%s", predicate_table->symbols[pred]);
  if (ar > 0) {
    if (IS_NUMBER(it[0]))
      pos += sprintf(&buf[pos], "(%ld", GET_VALUE(it[0]));
    else
      pos += sprintf(&buf[pos], "(%s", constant_table->symbols[it[0]]);
    for ( i = 1; i < ar; i++)
      if (IS_NUMBER(it[i]))
	pos += sprintf(&buf[pos], ",%ld", GET_VALUE(it[i]));
      else
	pos += sprintf(&buf[pos], ",%s", constant_table->symbols[it[i]]);
    pos += sprintf(&buf[pos], ")");
  }
  buf[pos] = '\0';
  if (pos > BUFSIZ) {
    fprintf(stderr, "buffer overflow -- core dumped\n");
    abort();
  }
  return  buf;
}
  

static long print_text_instance(Instance *it, long pred, int ar) 
{
  static FILE *ofile = sys_data.output_file;
  char *buf = NULL;
  long atom = -1;

  buf = get_ground_instance(it, pred, ar);

  if ((atom = atom_table->Lookup(buf)) < 0) {
    //insert the atom to the table
    atom = atom_table->Insert(buf);
    sys_data.ground_atoms++;
  }
  if (!buf)
    int_error("NULL ground instance", "");
  
  fprintf(ofile, buf);
  return atom;
}

static long print_smodels_instance(Instance *it, long pred, int ar)
{
  char *buf = NULL;
  long atom = -1;
  
  buf = get_ground_instance(it, pred, ar);

  if ((atom = atom_table->Lookup(buf)) < 0) {
    //insert the atom to the table
    atom = atom_table->Insert(buf);
    sys_data.ground_atoms++;
  }

  if (atom < 0)
    int_error("cannot create ground atom '%s'", buf);

  fprintf(sys_data.output_file, "%ld ", atom+1);
  return atom;
}

long print_instance(Instance *it, long pred, int ar)
{
  if (sys_data.emit_text)
    return print_text_instance(it, pred, ar);
  else
    return print_smodels_instance(it, pred, ar);
}

inline int equal_item(Instance *i1, Instance *i2, int arity)
{
  int i;
  for (i = 0; i < arity; i++)
    if (i1[i] != i2[i])
      return 0;
  return 1;
}


HashSet::HashSet(long sz, int dim, long bs)
{
  long i;
  max_size = 0;
  while (primes[max_size] <= sz)
    max_size++;
  size = 0;
  arity = dim;
  if (arity <= 0)
    max_arity = 1;
  else
    max_arity = 0;

  if (bs != 0)
    base = bs;
  else
    base = BASE_DEFAULT;
  tmp_items = NULL;
#ifdef DEBUG
  probes = new long[MAX_PROBE];
  if (!probes)
    error(SYS_ERR,"malloc error");
  for (i = 0; i < MAX_PROBE; i++)
    probes[i] = 0;
#endif
  debug(DBG_INSTANCE, 1, "Creating HashSet");
  items = new Instance*[primes[max_size]];
  valid = new char[primes[max_size]];
  if (!items || !valid)
    error(SYS_ERR,"malloc error", primes[max_size]);
  memset(valid, 0, primes[max_size]);
  for (i = 0; i < primes[max_size]; i++) {
    items[i] = new Instance[max_arity];
  }
}
  
HashSet::~HashSet()
{
  long  i;
  debug(DBG_INSTANCE, 1, "Deleting HashSet");
#ifdef DEBUG
  delete [] probes;
#endif
  for (i = 0; i < primes[max_size]; i++)
    delete [] items[i];
  delete [] items;
  delete [] valid;
}

// Memory is not usually freed when clearing a HashSet but is resize
// instead. This is why the maximum arity is stored. If the new arity
// is less than or equal to maximum arity the same arrays can be
// resize, otherwise new arrays must be allocated. The values are not
// actually cleared but the valid-bit is set to 0. 
void HashSet::Clear(int new_ar, long new_base)
{
  debug(DBG_INSTANCE, 1, "Clearing HashSet");
  debug(DBG_INSTANCE, 4, "\tArity: %d -> %d", arity, new_ar);
  debug(DBG_INSTANCE, 4, "\tBase: %d -> %d", base, new_base);
  long i;
  memset(valid, 0, primes[max_size]);

  if (new_ar > max_arity) {
    for (i = 0; i < primes[max_size]; i++) {
      delete [] items[i];
      items[i] = new Instance[new_ar];
    }
    max_arity = new_ar;
  }
  base = new_base;
  arity = new_ar;
  size = 0;
}



unsigned long HashSet::Hash(Instance *key, int sz)
{
  debug(DBG_INSTANCE, 3, "HashSet::Hash (%ld)", key);
  unsigned long val = 0, b = 0;
  Instance **tbl = NULL;
  char *v;
  long int i;

#ifdef DEBUG
  long pr = 0;
#endif

  if (sz) {
    tbl = tmp_items;
    v = new_valid;
  } else {
    tbl = items;
    v = valid;
  }
  for (i = 0; i < arity; i++) {
    b *= base;
    b += key[i];
    b %= primes[max_size+sz];
  }

  for (i = 0; i < primes[max_size+sz]; i++) {
    val = b + i*i;
    val %= primes[max_size+sz];

    debug(DBG_INSTANCE, 5, "Probing (%s): %d", key, val);

#ifdef DEBUG
    pr++;
    if (pr == MAX_PROBE)
      pr--;
#endif
    if (!v[val]) {
#ifdef DEBUG
      probes[pr]++;
#endif
      return val;
    }
    else if (equal_item(key, tbl[val], arity)) {
#ifdef DEBUG
      probes[pr]++;
#endif
      return val;
    }
  }

  // NOT FOUND
  return primes[max_size+sz]+1;
}
    
long HashSet::Insert(Instance *key)
{
  long index = 0, i = 0;
  Instance *new_node = NULL;
  debug(DBG_INSTANCE, 2, "Inserting value (%s, %d)", key, arity);
  index = Hash(key);

  assert(index < primes[max_size]);
  // Is it in set already?
  if (valid[index]) {
    return index;
  }

  // Mark the item valid
  valid[index] = 1;
  new_node = items[index];
  
  if (!new_node)
    error(SYS_ERR,"malloc error");
  
  for (i = 0; i < arity; i++)
    new_node[i] = key[i];
  
  items[index] = new_node;
  size++;
  if ( size > primes[max_size]/2) {
    Rehash();
    // calculate new index
    index = Hash(key);
  }
  return index;
}

long HashSet::Lookup(Instance *key)
{
  long index = Hash(key);

  assert( index < primes[max_size]);
  if (valid[index])
    return 1;
  else
    return 0;
}
  
void HashSet::Rehash()
{
 long new_max_size = max_size+1, i, index = 0;
 debug(DBG_SYMBOL, 3, "Rehashing HashSet:\n\tOld max_size: %d\n\tNew"
       "max_size: %d\n", primes[max_size], primes[new_max_size]);
 
 tmp_items = new Instance *[primes[new_max_size]];

 new_valid = new char[primes[new_max_size]];
 memset(new_valid, 0, primes[new_max_size]);
 if (!tmp_items)
   error(SYS_ERR,"malloc error");
 
 memset(tmp_items, 0, sizeof(Instance*)*primes[new_max_size]);
 for (i = 0; i < primes[max_size]; i++) {
   if (valid[i]) {
     index = Hash(items[i], 1);
     assert( index < primes[new_max_size]);
     tmp_items[index] = items[i];
     new_valid[index] = 1;
   } 
 }
 for (i = 0; i < primes[max_size]; i++) {
   if (!valid[i])
     delete [] items[i];
 }
 for (i = 0; i < primes[new_max_size]; i++) {
   if (!new_valid[i])
     tmp_items[i] = new Instance[max_arity];
 }
 max_size = new_max_size;
 debug(DBG_INSTANCE, 4, "Rehash: Deleting old nodes");
 delete [] items;
 delete valid;
 valid = new_valid;
 items = tmp_items;
}
  
    

RBTree::RBTree(int ar, int in)
{
  head = x =   new TreeNode;
  arity = ar;
  index = in;
  if (!head)
    error(SYS_ERR,"malloc error");
  z = new TreeNode;
  z->left = z->right = z->next = z;
  z->red = 0;
  z->item = new Instance[ar];
  z->item[index] = 0;
  head->right = head->left = head->next = z;
  head->item = new Instance[ar];
  if (!head->item)
    error(SYS_ERR,"malloc error");
  memset(head->item, -1, ar * sizeof(Instance));
}

RBTree::~RBTree()
{
  delete head;
}

int RBTree::Lookup(Instance *item)
{
  Instance key = item[index];
  TreeNode *tmp = head;

  while (tmp != z) {
    if (tmp->item[index] < key)
      tmp = tmp->right;
    else if (tmp->item[index] > key)
      tmp = tmp->left;
    else if (equal_item(item, tmp->item, arity))
      return 1;
    else // kaikki samanarvoiset talla tasolla
      tmp = tmp->next;
  }
  return 0;
}


void RBTree::Insert(Instance *item)
{
  TreeNode *new_node;
  Instance v = item[index];
  x = head, p = head, g = head, gg = head;
  
  while (x != z) {
    gg = g; g = p; p = x;

    if (v < x->item[index])
      x = x->left;
    else if (v > x->item[index])
      x = x->right;
    else { // loytyi samanarvoinen, lisataan siihen
      new_node = new TreeNode;
      new_node->item = item;
      new_node->red = 0;
      new_node->left = new_node->right = z;
      new_node->next = x->next;
      x->next = new_node;
      return;
    }
    // onko kyseessa 4-node
    if (x->left->red &&	x->right->red)
      Split(v);
  }
  x = new TreeNode;
  x->item = item;
  x->red = 0;
  x->left = x->right = x->next = z;

  if (v < p->item[index])
    p->left = x;
  else
    p->right = x;
  Split(v);
}


void RBTree::Split(Instance v)
{
  x->red = 1;
  x->left->red = 0;
  x->right->red = 0;
  if (p->red) {
    g->red = 1;
    if ((v < g->item[index]) != (v < p->item[index]))
      p = Rotate(v,g);
    x = Rotate(v,gg);
    x->red = 0;
  }
  head->right->red = 0;
}

TreeNode *RBTree::Rotate(Instance v, TreeNode *y)
{
  TreeNode *c, *gc;
  c = (v < y->item[index]) ? y->left : y->right;
  if (v < c->item[index]) {
    gc = c->left;
    c->left = gc->right;
    gc->right = c;
  } else {
    gc = c->right;
    c->right = gc->left;
    gc->left = c;
  }
  if ( v < y->item[index])
    y->left = gc;
  else
    y->right = gc;
  return gc;
}

#define IND(x) {for(i=0;i < x;i++) printf(" "); }

void RBTree::Print(TreeNode *tn, int id)
{
  int i;
  if (tn == z)
    return;
  IND(id);
  printf("Color: %s\n", (tn->red ? "red" : "black"));
  IND(id);
  printf("Key: %ld\n", tn->item[index]);
  IND(id);
  printf("Left:\n");
  Print(tn->left, id +2);
  IND(id);
  printf("Right:\n");
  Print(tn->right, id +2);
  IND(id);
  printf("Next:\n");
  Print(tn->next, id +2);
}

