/*
  graph.cc -- implementation of dependency graphs for lparse

  This program has no warranties of any kind. Use at own risk.
  
  Author: Tommi Syrjnen (tommi.syrjanen@hut.fi)
  
  $Id: graph.cc,v 1.1 1998/08/04 09:19:36 tssyrjan Exp $	 
  */

#include "predicate.h"
#include "global.h"
#include "graph.h"
#include "debug.h"
#include <string.h>
#include "array.h"

static char *domain_string[] = { "DM_UNKNOWN", "DM_FALSE", "DM_TRUE",
				 "DM_DOMAIN" }; 

GraphNode::GraphNode()
  : children(NULL)
{
  size = 0;
  children = new ResizableArray();
  if (!children)
    error(SYS_ERR, "malloc error");
}

void GraphNode::AddChild(long child)
{
  (*(*children)[size++]) = child;
}


Graph::Graph()
{
  size = 0; visited = NULL; 
  domain = NULL;
  nodes = new GraphNodeArray();
  if (!nodes)
    error(SYS_ERR, "malloc error");
}

Graph::~Graph()
{
  delete nodes;
  if (domain)
    delete [] domain;
  domain = NULL;
} 

void Graph::AddNode(long nd)
{
  (*nodes)[nd]->index = size++;
  
  /* delete old information */
  if (domain) {
    delete [] domain;
    domain = NULL;
  }
}

void Graph::AddEdge(long x, long y)
{
  /* sanity check */
  if ( (x > size) || (x < 0) || (y > size) || (y < 0))
    int_error("invalid edge in AddEdge", x);

  (*nodes)[x]->AddChild(y);
}

void Graph::CheckCycles()
{
  long i;
  debug(DBG_GRAPH, 2, "Finding cycles in graph");
  debug(DBG_GRAPH, 4, "Creating tables");

  visited = new long[size+1];

  if (!domain)
    domain = new DomainType[size+1];

  if (!domain || !visited)
    error(SYS_ERR, "malloc error");

  // initialize domain and visited information 
  for (i = 0; i <= size; i++) {
    domain[i] = DM_UNKNOWN;
    visited[i] = 0;
  }  

  // visit every node 
  for (i = 0; i <= size; i++) 
    if (domain[i] == DM_UNKNOWN)
      Visit(i);
  return;
}


int Graph::Visit(long node)
{
  long i, val = 0, chld = 0;
  debug(DBG_GRAPH, 4, "Visiting node %d: %s", node, predicates[node]->Name());

  // mark this node as visited
  visited[node] = 1;
  
  // suppose by default that node is a domain predicate
  domain[node] = DM_TRUE;

  // now visit each children of the node
  for (i = 0; i < (*nodes)[node]->size; i++) {
    chld = *((*(*nodes)[node]->children)[i]); // ugly
    debug(DBG_GRAPH, 5, "\tChild %d", chld);
    // if we have visited the child previously a cycle is found 
    if ( visited[chld] == 1 )
      domain[node] = DM_FALSE;
    // if we haven't processed the child before process it, otherwise
    // use the calculated value.
    else if ( domain[chld] == DM_UNKNOWN)
      val = Visit(chld);
    else
      val = domain[chld];

    debug(DBG_GRAPH, 5, "\t\tDomain %d", domain[chld]);

    if (val == DM_FALSE)
      domain[node] = DM_FALSE;
  }

  // clear the visited mark of the node
  visited[node] = 0;

  // if the node is a domain predicate, calculate its domain.
  if (domain[node] == DM_TRUE) 
    predicates[node]->CalculateDomain();

  // return the domain value of the node.
  return (int) domain[node];
}

// prints the domain values. For debugging only.
void Graph::PrintDomains()
{
  long i;
  if (!domain)
    return;

  for (i = 0; i < size; i++) {
    printf("Node %ld: %s\n", i, domain_string[domain[i]]);
  }

  return;
}

