/* symbol.h -- definition of symbol table structures

   Defines classes:
   	SymbolTable	-- well, a normal symbol table
	FunctionTable	-- for storing pointers to functions
   
   This program has no warranties of any kind. Use at own risk.

   Author: Tommi Syrjnen (tommi.syrjanen@hut.fi)

   $Id: symbol.h,v 1.1 1998/08/04 09:18:18 tssyrjan Exp $	 
*/

#ifndef SYMBOL_H
#define SYMBOL_H

#include "debug.h"
#include "defines.h"

// Normal identifier, only value is needed. 
// should be coded in predicate definition: like foo(X,Y) -> foo@2 
struct SymbolNode
{
  char *symbol;
  long value;
};

/* Class Symboltable is a closed automatically resizing hash table. It
   is used for storing predicates, variables and nonnumeric
   constants. 

   Attributes:
   	SymbolNode **items	-- the actual table
	SymbolNode **tmp_items	-- temporary table used in resizing
	long size		-- how many items are stored in table
	long max_size		-- How big the table is.
	***NOTE: max_size is actually an index to primes array. If you
	   try to use it directly you will be badly hosed. (Been
	   there, done that, trust me) ***
        long max_arity		-- What is the greatest arity seen
			           this far. This is meaninful only
				   for predicates and it is used by
				   functions that want to know how
				   much to reserve memory for each
				   instance.
	char **symbols		-- Array of all symbols in program
				   indexed by their values. 
   Methods:

   	long Insert(char *key, int ar = -1)
		-- Inserts symbol 'key' with arity 'ar' to
		   table. Returns the value of symbol. Removes
		   duplicates.
        long Lookup(char *key)	-- returns value of 'key' or -1 if not
				   found
	char *LookupByValue(long val)
		-- Returns symbol corresponding to 'val' You shouldn't
		   modify the returned string since it points to
		   actual entry in the table. 
	void CreateSymbolArray() -- Initializes 'symbols' array.
	long Size()		-- returns number of items in table
	int MaxArity()		-- returns biggest arity seen
	long Hash(char *key, int sz = 0)
		-- Finds hash value for 'key'. Sz should be '1' when
		   rehashing and '0' otherwise.
	void ReHash()		-- approximately doubles the size of
				   the hash table
	void PrintItems()	-- Prints out all the items in hash table
   */
class SymbolTable
{
public:
  SymbolTable(long sz = SYMBOL_DEFAULT_SIZE);  
  ~SymbolTable();
  
  long Insert(char *key, int arity = -1);
  long Lookup(char *key, int arity = -1);
  inline char *LookupByValue(long val);
  long Size() { return size; }
  int MaxArity() { return max_arity; }
  void CreateSymbolArray();
#ifdef DEBUG
  void PrintItems();
#endif
  char **symbols;

private:
  unsigned long Hash(char *key, int sz = 0);
  void ReHash();

  SymbolNode **items, **tmp_nodes;

  long size, max_size, max_arity;
};


  
  

// For declaring functions the function to be called must be specified.
struct FunctionNode
{
  char *symbol;
  int valid; // has the function been inserted
  // pointer to function with at least one long-argument.
  InstFunc func;
};

/* This is like SymbolTable with few exceptions:
   FunctionTable doesn't have symbols array and max_arity. 

   At the beginning of the program the external library is processed
   and all library functions are registered to FunctionTable. That is,
   they are inserted into the table and 'valid' bit is set on. Later
   only functions that have been registered can be inserted into
   table. Otherwise Insert returns -1. The caller must check this or
   wery strange things happen when constructing the function call
   trees. 

   Also, the function bindings can be changed or added with Define()
   function call. Both the symbol to be defined doesn't have to be
   registered but the target function has to be.    
   */

class FunctionTable
{
public:
  FunctionTable(long sz = SYMBOL_DEFAULT_SIZE);  
  ~FunctionTable();
  
  long Insert(char *key);
  long Register(char *key, InstFunc p, int val = 0); 
  long Lookup(char *key);
  long Define(char *symbol, char *value, int val = 0);

  InstFunc GetFunction(char *key);
  long Size() { return size; }
  void PrintRegistered();
#ifdef DEBUG
  void PrintItems();
#endif

private:
  unsigned long Hash(char *key, int sz = 0);
  void ReHash();

  FunctionNode **items, **tmp_nodes;
  long size, max_size;
};

#endif
