/* predicate.cc -- implementation of predicate data type 
      
   This program has no warranties of any kind. Use at own risk.

   Author: Tommi Syrjnen (tommi.syrjanen@hut.fi)

   $Id: predicate.cc,v 1.1 1998/08/04 09:19:05 tssyrjan Exp $	 
*/

#include "predicate.h"
#include "symbol.h"
#include <string.h>
#include "iterate.h"

Predicate::Predicate()
  : indices(NULL), atoms(NULL), existing_indexes(0), rules(UNORDERED),
    it(NULL), name(NULL), arity(-1), status(DM_UNKNOWN)
{
  pred = -1;
  complement = -1;
  emitted = NULL;
}


Predicate::~Predicate()
{
#if 0
  int i;
  for (i = 0; i < arity; i++) {
    if (indices[i])
      delete indices[i];
  } 
  delete [] indices;
  delete atoms;
#endif
}

RestrictType Predicate::CheckRestrict()
{
  debug(DBG_RESTRICT, 2, "Checking restrict for %s", name);
  assert(arity >= 0);
  RestrictType result = RT_STRONG, tmp = RT_STRONG;
  Rule *rl = NULL;
  
  while ((rl = rules.Iterate())) {
    tmp = rl->CheckRestrict();
    switch (tmp) {
    case RT_NONE:
      result = RT_NONE;
      break;
    case RT_WEAK:
      if (result > RT_NONE)
	result = RT_WEAK;
      break;
    default:
      break;
    }
  }
  debug(DBG_RESTRICT, 2, "\tResult: $d", result);
  return result;
}

void Predicate::AddRule(Rule *rl)
{
  assert(arity >= 0);
  // check if the rule is a fact
  if ((rl->positive.Size() == 0) && (rl->negative.Size() == 0)) {
    AddInstance(rl->head->cons);
  } else
    rules.Insert(rl);
}

void Predicate::AddInstance(Instance *it)
{
  assert(arity > 0);
  atoms->Insert(it);
}
    
void Predicate::SetArity(int ar)
{
  arity = ar;
  if (!atoms) {
    atoms = new InstanceSet(sys_data.domain_size, ar, BASE_DEFAULT);
    it = new InstanceIterator(atoms);
    if (!atoms || ! it)
      error(SYS_ERR, "malloc error");
  } else {
    atoms->Clear(ar, BASE_DEFAULT);
    it->Clear();
  }
}
  
void Predicate::CreateIndex(int ind)
{
  Instance *item = NULL;
  assert((ind >= 0) && (ind < arity));

  if (!indices) {
    indices = new Index*[arity];
    if (!indices)
      error(SYS_ERR, "malloc error");
    memset(indices, 0, sizeof(Index*)*arity);
  }
  if (existing_indexes & (1 << ind)) 
    int_error("trying to create same index two times in predicate"
	      " '%s'", name);

  indices[ind] = new Index(arity, ind);
  if (!indices[ind])
    error(SYS_ERR, "malloc error");

  existing_indexes |= (1 << ind);

  it->Clear();
  while ((item = it->Iterate())) 
    indices[ind]->Insert(item);
}


void Predicate::Emit()
{
  Instance *item = NULL;
  Rule *rl = NULL;
  Instance atom = -1;

  // check if this is a domain predicate that shouldn't be printed
  if (((sys_data.print_domains == PR_NONE) &&
       (status == DM_DOMAIN)) || (status == DM_INTERNAL))
    return;
  it->Clear();
  
  if (arity == 0) {
    sys_data.ground_atoms++;
  }
  // print the items
  while ((item = it->Iterate())) {
    atom = print_instance(item, pred, arity);
    fprintf(sys_data.output_file, "%s\n", (sys_data.emit_text) ?
	    "." :"0 0");
    sys_data.ground_rules++;
    sys_data.ground_sigma++;
    if (complement >= 0) { // print the complementary rule
      if (!emitted->Lookup(&atom)) {
	sys_data.ground_rules++;
	false_lit->EmitGround();
	emitted->Insert(&atom);
	if (sys_data.emit_text)
	  fprintf(sys_data.output_file, " :- ");
	else
	  fprintf(sys_data.output_file, " 0 2 ");
	print_instance(item, pred, arity);
	if (sys_data.emit_text)
	  fprintf(sys_data.output_file, ", ");
	else
	  fprintf(sys_data.output_file, " ");
	atom = print_instance(item, complement, arity);
	predicates[complement]->emitted->Insert(&atom);
	if (sys_data.emit_text)
	  fprintf(sys_data.output_file, ".\n");
	else
	  fprintf(sys_data.output_file, "\n");
	sys_data.ground_sigma += 2;
      }
    }
  }
  // and the rules
  if (status != DM_DOMAIN) {
    while ((rl = rules.Iterate()))
      rl->GroundRule(0);
  }
}
void Predicate::CalculateDomain()
{
  debug(DBG_GROUND, 2, "Calculating domain for %s", name);
  Rule *rl = NULL;

  // ground all rules one at time
  while ((rl = rules.Iterate())) {
    rl->GroundRule(1);
  }
  // don't change status of internal range predicates
  if (status != DM_INTERNAL)
    SetStatus(DM_DOMAIN);
}
      

void Predicate::EmitAll()
{
  long i = 0;

 
  for( i = 0; i < predicate_table->Size(); i++)
    predicates[i]->Emit();
}


RestrictType Predicate::CheckAllRestricts()
{
  RestrictType rt = RT_STRONG, tmp = RT_STRONG;
  long i = 0;
  
  for( i = 0; i < predicate_table->Size(); i++){
    tmp = predicates[i]->CheckRestrict();
    switch (tmp) {
    case RT_NONE:
      rt = RT_NONE;
      break;
    case RT_WEAK:
      if (rt > RT_WEAK)
	rt = RT_WEAK;
      break;
    default:
      break;
    }
  }
  return rt;
}
	
void Predicate::MakeComplement(long p1, char *cmp, int ar)
{
  assert( (p1 >= 0) && cmp);
  long p2 = -1;

  if (predicates[p1]->complement >= 0)
    return;
  
  // initialize the complementary predicate if needed 
  p2 = predicate_table->Insert(cmp, ar);
  if (p2 < 0)
    int_error("unknown p2icate '%s'", cmp);
  if (predicates[p2]->Pred() != p2) {
    dependency_graph->AddNode(p2);
    predicates[p2]->SetName(cmp);
    predicates[p2]->SetPred(p2);
    predicates[p2]->SetArity(ar);
  }    
  predicates[p1]->complement = p2;
  predicates[p1]->emitted = new InstanceSet(DOMAIN_DEFAULT_SIZE, 1,
					    BASE_DEFAULT);
  predicates[p2]->complement = p1;
  predicates[p2]->emitted = new InstanceSet(DOMAIN_DEFAULT_SIZE, 1,
					    BASE_DEFAULT);;
  if (!predicates[p1]->emitted || !predicates[p1]->emitted)
    error(SYS_ERR, "malloc error");
}
