/* literal.cc -- implementation of literals
      
   This program has no warranties of any kind. Use at own risk.

   Author: Tommi Syrjnen (tommi.syrjanen@hut.fi)

   $Id: literal.cc,v 1.1 1998/08/04 09:19:17 tssyrjan Exp $	 
*/

#include "debug.h"
#include "literal.h"
#include "global.h"
#include "parse.h"
#include "instance.h"
#include <stdio.h>
#include "symbol.h"
#include "array.h"
#include "predicate.h"

Literal::Literal(ParseNode *nd, long p, int ar)
{
  ParseNode *tmp = NULL;
  int pos = ar -1; // for reverse travelsal
  Term *tm = NULL;
  ground = 1;
  has_range = 0;
  has_function = 0;
  negative = ( nd->type == PT_NEG_ATOM );
  arity = ar;
  numvars = 0;
  pred = p;

  if (arity > 0) {
    args = new TermType[arity];
    vars = new long[arity];
    cons = new Instance[arity];
    terms = new Term*[arity];
    if (!args || !vars || !cons || !terms)
      error(SYS_ERR, "malloc error");
  } else {
    args = NULL;
    vars =  NULL;
    cons =  NULL;
    terms = NULL;
  }
  tmp = nd->right;
  // check the arguments
  while (tmp) {
    if (!tmp->left)
      int_error("missing argument", "");

    tm = Term::ConstructArgument(tmp->left);
    if (!tm)
      int_error("null term", "");
    switch(tmp->left->type) {
    case PT_VARIABLE: 
      numvars++;
      ground = 0;
      args[pos] = T_VARIABLE;
      vars[pos] = tm->val;
      cons[pos] = -1;
      terms[pos] = NULL;
      delete tm;
      break;
    case PT_NUMBER:
    case PT_CONSTANT:
      args[pos] = T_CONSTANT;
      vars[pos] = -1;
      cons[pos] = tm->val;
      terms[pos] = NULL;
      delete tm;
      break;
    case PT_FUNCTION:
    case PT_EXPR:
      args[pos] = T_FUNCTION;
      has_function = 1;
      vars[pos] = -1;
      cons[pos] = -1;
      terms[pos] = tm;
      if (!tm->ground)
	ground = 0;
      break;
    case PT_RANGE:
      has_range = 1;
      args[pos] = T_RANGE;
      vars[pos] = -1;
      cons[pos] = -1;
      terms[pos] = tm;
      break;
    default:
      int_error("misformed parse tree: invalid literal argument type"
		"'%s'", parse_strings[tmp->left->type]);
    }
    tmp = tmp->right;
    pos--;
  }
}

Literal::Literal(Variable v, long p )
{
  arity = 1;
  numvars = 1;
  ground = 0;
  negative = 0;
  has_range = 0;
  pred = p;

  args = new TermType;
  vars = new Variable;
  cons = new Instance;
  terms = new Term*;

  if (!args || !vars || !cons  || !terms)
    error(SYS_ERR, "malloc error");

  *args = T_VARIABLE;
  *vars = v;
  *cons = -1;
  *terms = NULL;
}

Literal::Literal(Variable *v, long p, int ar )
{
  int i = 0;
  arity = ar;
  numvars = ar;
  ground = 0;
  negative = 0;
  has_range = 0;
  pred = p;

  args = new TermType[ar];
  vars = new Variable[ar];
  cons = new Instance[ar];
  terms = new Term*[ar];

  if (!args || !vars || !cons  || !terms)
    error(SYS_ERR, "malloc error");

  for (i = 0; i < arity; i++) {
    args[i] = T_VARIABLE;
    vars[i] = v[i];
    cons[i] = -1;
    terms[i] = NULL;
  }
}



Literal::Literal()
{
  arity = 0;
  numvars = 0;
  ground = 0;
  negative = 0;
  has_range = 0;
  pred = 0;
}

int Literal::Test()
{
  int i = 0;
  static Instance *item = NULL;

  if (!item) {
    item = new Instance[arity];
    if (!item)
      error(SYS_ERR, "malloc error");
  }
  
  for (i = 0; i < arity; i++) {
    switch (args[i]) {
    case T_VARIABLE:
      item[i] = variables[vars[i]];
      break;
    case T_CONSTANT:
      item[i] = cons[i];
      break;
    default:
      int_error("misformed literal: invalid argument type '%d'",
		args[i]);
    }
  }
  if (predicates[pred]->atoms->Lookup(item))
    return 0;
  else
    return 1;
}
    
int Literal::GetPos()
{
  int res = 0, i = 0;

  for (i = 0 ; i < arity; i++) {
    if (args[i] == T_VARIABLE)
      res = max(res, variables[vars[i]]);
  }

  return res;
}
  
Instance Literal::EmitGround()
{
  static Instance *item = NULL;
  int i = 0; 

  if (!item) {
    item = new Instance[arity];
    if (!item)
      error(SYS_ERR, "malloc error");
  }
  sys_data.ground_sigma++;
  for (i = 0; i < arity; i++) {
    switch (args[i]) {
    case T_VARIABLE:
      item[i] = variables[vars[i]];
      break;
    case T_CONSTANT:
      item[i] = cons[i];
      break;
    default:
      int_error("invalid term type '%d' in Literal::EmitGround()",
		args[i]);
    }
  }
  return print_instance(item, pred, arity);
}

Instance Literal::EmitComplement()
{
  static Instance *item = NULL;
  int i = 0; 

  if (!item) {
    item = new Instance[arity];
    if (!item)
      error(SYS_ERR, "malloc error");
  }
  sys_data.ground_sigma++;
  for (i = 0; i < arity; i++) {
    switch (args[i]) {
    case T_VARIABLE:
      item[i] = variables[vars[i]];
      break;
    case T_CONSTANT:
      item[i] = cons[i];
      break;
    default:
      int_error("invalid term type '%d' in Literal::EmitGround()",
		args[i]);
    }
  }
  return print_instance(item, predicates[pred]->complement, arity);
}




void Literal::Restrict(int t)
{
  int i = 0;

  for (i = 0; i < arity; i++) {
    if ((vars[i] >= 0) && (variables[vars[i]] < t))
      variables[vars[i]] = t;
  }
}

void Literal::CreateInstance()
{
  int i;
  static Instance *it = NULL;

  if (! it) {
    it = new Instance[arity];
    if (! it)
      error(SYS_ERR, "malloc error");
  }

  for (i = 0; i < arity; i++) {
    if (args[i] == T_VARIABLE) 
      it[i] = variables[vars[i]];
    else if (args[i] == T_CONSTANT)
      it[i] = cons[i];
    else
      int_error("misformed literal", "");
  }
  predicates[pred]->AddInstance(it);
}

void Literal::Print()
{
  char *buf = GetPrintString();
  fprintf(stderr, "%s", buf);
}
char *Literal::GetPrintString()
{
  char buf[BUFSIZ] = { 0 };
  int pos = 0, i = 0;
  int first = 1;
  Range *r = NULL;

  pos += sprintf(&buf[pos], "%s", predicate_table->symbols[pred]);
  if (arity > 0) {
    pos += sprintf(&buf[pos], "(");
    for (i = 0; i < arity; i++) {
      if (!first)
	pos += sprintf(&buf[pos], ",");
      first = 0;
      
      switch (args[i]) {
      case T_VARIABLE:
	pos += sprintf(&buf[pos], "%s", variable_table->symbols[vars[i]]);
	break;
      case T_CONSTANT:
	if (IS_NUMBER(cons[i]))
	  pos += sprintf(&buf[pos], "%ld", GET_VALUE(cons[i]));
	else
	  pos += sprintf(&buf[pos], "%s", constant_table->symbols[cons[i]]);
	break;
      case T_FUNCTION:
	((Function*)terms[i])->Print();
	break;
      case T_RANGE:
	r = (Range *) terms[i];
	pos += sprintf(&buf[pos], "%ld..%ld", r->start, r->end);
	break;
      default:
	int_error("invalid term type '%d' in literal", args[0]);
      }
    }
    pos += sprintf(&buf[pos], ")");
  }
  buf[pos] = '\0';
  if (pos >= BUFSIZ){
    fprintf(stderr, "buffer overflow -- core dumped");
    abort();
  }
  return buf;
}
 
