/*
  list.h -- list classes for lparse
  This program has no warranties of any kind. Use at own risk.
  
  Author: Tommi Syrjnen (tommi.syrjanen@hut.fi)
  
  $Id: list.h,v 1.1 1998/08/04 09:18:33 tssyrjan Exp $	 
*/
#ifndef LIST_H
#define LIST_H

enum ListType { UNORDERED, ORDERED_BIG, ORDERED_SMALL };

struct ListNode {
  long weight;
  void *item;
  ListNode *next;
};

/*
 * List is an abstract class to be used as a base class and actual
 * lists for each type will be inherited from it. It would be more
 * natural to use templates, but there seems to be some problems with
 * template instantiation. Yup. Pretty ugly. 
 *
 * List stores (void *) pointers, and the derived classes should act
 * as wrappers converting items to and from void pointers.
 *
 * The type of the list (ordered or unordered) is chosen by argument
 * to constructor.
 *
 * List uses sentinel node at the start of the list.
 *
 */
class List
{
public:
  // Constructor for List. 
  List(ListType type = UNORDERED);
  ~List();
  
  /**
   * Inserts an item to list. Inserting an item resets iteration
   * counter. Weight defaults as 0 so the same function can be used
   * for both ordered and unordered lists.
   */
  void Insert(void *item, long weight = 0);
  void Append(void *item, long weight = 0);
  
  /// Returns the items one at time. Returns NULL at the end. 
  void *Iterate();

  /** Merges another with this list. Same weighting duplicates are
   * removed for ordered lists, but not for unordered.
   */
  void Merge(List *lst);

  void Clear();

  int Find(void *item);
  void ClearIterator() { current = items->next; }
  
  long Length() {return size; }
  long Size() { return size; }
  ListType Type() { return type; }
protected:
  void MergeUnordered(List *lst);
  void MergeOrdered(List *lst);

  ListNode *CreateEmptyNode();
  ListNode *DuplicateNode(ListNode *);

  ListNode *items, *current, *last;
  long size, last_weigth;
  ListType type;
};


class Rule;

/**
 * RuleList implements a list of rules.
 */
class RuleList : public List
{
public:
  RuleList(ListType type = UNORDERED);
  ~RuleList();

  void Insert(Rule *item, long weight = 0);
  void Append(Rule *item, long weight = 0);
  Rule *Iterate();
};

class Literal;

class LiteralList : public List
{
public:
  LiteralList(ListType type = UNORDERED);
  ~LiteralList();

  void Insert(Literal *item, long weight = 0);
  Literal *Iterate();
};

class LongList : public List
{
public:
  LongList(ListType type = UNORDERED);
  ~LongList();

  void Insert(long item, long weight = 0);
  long Iterate();
};


class Function;

class FunctionList : public List
{
public:
  FunctionList(ListType type = UNORDERED);
  ~FunctionList();

  void Insert(Function *item, long weight = 0);
  Function *Iterate();
};


#endif

  
