#!/bin/bash

# Translator for combining ASP and linear constraints (LC) over integers
# based on the approach described in the KR 2012 paper:
#
# T. Janhunen, G. Liu, and I. Niemela: Answer Set Programming
# via Mixed Integer Programming.
#
# (c) 2011 Tomi Janhunen

DIR=.        # Consider replacing this by the absolute path (pwd)
BIN=$DIR/bin
DEF=$DIR/def
TMP=$DIR/tmp

# USAGE: mingo.sh [--ground-only] <gringo arguments>

# =============================================================================
# Local Functions

function cleanup()
{
  for f in $purge
  do
    rm -f $f
  done
}

function toclean()
{
  if [ $# -gt 0 ]
  then
    touch $*
    purge=$purge" "$*
    trap cleanup EXIT
  fi
}

# =============================================================================

ground=$TMP/$$-ground.sm
symfile=$TMP/$$-symbols.sm
tmpfile=$TMP/$$-file.s
translation=$TMP/$$-translation.lp
theory=$TMP/$$-theory.lp
model=$TMP/$$-model.txt

# Declare temporary files for removal upon exit

toclean $ground $symfile $tmpfile $translation $theory $model

# OPTIONS

if test "$1" = "--ground-only"
then
  let gnd=1
  shift
else
  let gnd=0
fi

# GROUND

$BIN/gringo $* $DEF/ext.lp \
| $BIN/lpcat -m -s=$symfile > $ground 2>> /dev/null
if test $? -ne 0
then
  echo "$0: grounding '""$*""' failed!"
  exit -1
fi

# EXTRACT SYMBOLS

egrep -v -f $DEF/type-bin.egrep $symfile > $tmpfile
egrep -v -f $DEF/type-int.egrep $symfile > $tmpfile
rm -f $symfile
mv -f $tmpfile $symfile

# ASP TRANSLATION

$BIN/lp2mip $ground > $translation
let sp=`egrep -n "BINARY" $translation | sed 's/:BINARY//g'`-1
head -n +$sp $translation >> $theory

# THEORY ATOMS

read filter < $DEF/filter.egrep

egrep -h "$filter" $ground \
| $BIN/quote \
| sed 's/(/ /;s/)$//g' \
| while read n p r; do echo "$p($n,`echo $r | sed 's/ /,/g'`)"; done \
| cat $macros $DEF/expand.m4 - \
| m4 - \
| egrep -v '^$' >> $theory
if test $? -ge 2
then
  echo "$0: failed to translate theory atoms themselves!"
  exit -1
fi

# BOUND DEFINITIONS

echo "BOUNDS" >> $theory

egrep -f $DEF/type-int.egrep $ground \
| sed 's/[1-9][0-9]* //g' \
| sed 's/_/_u/g;s/(/_l/g;s/,/_c/g;s/)/_r/g' \
| sed 's/int_l/  /g;s/_r$/ free/g' \
| cat $DEF/expand.m4 - \
| m4 - \
| egrep -v '^$' >> $theory
if test $? -ge 2
then
  echo "$0: failed to handle bounds of integer theory variables!"
  exit -1
fi

# TYPE DEFINITIONS

echo "BINARY" >> $theory

egrep -f $DEF/type-bin.egrep $ground \
| sed 's/[1-9][0-9]* //g' \
| sed 's/_/_u/g;s/(/_l/g;s/,/_c/g;s/)/_r/g' \
| sed 's/bin_l/  /g;s/_r$//g' \
| cat $DEF/expand.m4 - \
| m4 - \
| egrep -v '^$' >> $theory
if test $? -ge 2
then
  echo "$0: failed to handle types of binary theory variables!"
  exit -1
fi

tail -n +`expr $sp + 2` $translation | head -n -1 >> $theory

egrep -f $DEF/type-int.egrep $ground \
| sed 's/[1-9][0-9]* //g' \
| sed 's/_/_u/g;s/(/_l/g;s/,/_c/g;s/)/_r/g' \
| sed 's/int_l/  /g;s/_r$//g' \
| cat $DEF/expand.m4 - \
| m4 - \
| egrep -v '^$' >> $theory
if test $? -ge 2
then
  echo "$0: failed to handle types of integer theory variables!"
  exit -1
fi

echo "END" >> $theory

# GROUND ONLY ?

if test $gnd -gt 0
then
  cat $theory
  exit 0
fi

# SOLVE

( echo "read "$theory; \
  echo "optimize"; \
  echo "display solution variables *" ) \
| $BIN/cplex > $model 2>/dev/null
if test $? -ne 0
then
  echo "$0: cplex did not accept the input!"
  exit -1
fi

# PRINT

if egrep "^MIP - Integer infeasible.$" $model 1>/dev/null
then
  echo "UNSATISFIABLE"
else
  if egrep "^Solution pool: [1-9][0-9]* solution[s]* saved.$" $model 1>/dev/null
  then
    echo "SATISFIABLE"
    fgrep "var_" $model \
    | fgrep "1.000000" \
    | sed 's/var_//g' \
    | sed 's/                         1.000000//g' \
    | $BIN/interpret $symfile \
    | egrep -v '^$'\
    | egrep -v '^'$filter'$' \
    | egrep -v -f $DEF/filter-types.egrep \
    | tr -d "\n"
    echo ""
    echo -n "Theory: "
    fgrep "var_" $model \
    | fgrep "1.000000" \
    | sed 's/var_//g' \
    | sed 's/[[:space:]]*1.000000//g' \
    | $BIN/interpret $symfile \
    | egrep -v '^$' \
    | egrep '^'$filter'$' \
    | tr -d "\n"
    echo ""
    echo -n "Vars: "
    egrep '^[[:lower:]].*[[:digit:]]+\.[[:digit:]]+$' $model \
    | egrep -v "^var_" \
    | sed 's/ /=/;s/ //g' \
    | sed 's/.000000//g' \
    | sed 's/_l/(/g;s/_r/)/g' \
    | tr "\n" " "
    echo "(other values 0)"
  else
    echo "UNKNOWN"
  fi
fi
