#! /bin/bash

#==============================================================================
# Configure the script.
#

# On some ubuntu versions, this directory is stored in memory.
TMP_DIR=/dev/shm

#==============================================================================
# Define helper functions.
#

function print_usage {
  cat <<EOF

Usage: $(basename $0 .sh) [--normalize-only] lp2normal [lp2normal-arg]... ";" clasp [clasp-arg]...

EOF
}

function clean_end {
  rm -f "$tmp_input" "$tmp_out" "$tmp_err"
  exit "$@"
}

#==============================================================================
# Parse and check arguments.
#

# Check for --normalize-only.
normalize_only=0
if [[ "$1" = --normalize-only ]]; then
  normalize_only=1
  shift
fi

# Grab the remaining command line arguments into a variable for easier handling.
declare -a args=("$@")

# Look for a semicolon ";" in the arguments.
semicolon_index=-1
for idx in "${!args[@]}"
do
  val="${args[idx]}"
  if [[ "$val" == ";" ]]; then
    semicolon_index="$idx"
    break
  fi
done

# It is an error if there is none.
if [[ "$semicolon_index" = -1 ]]; then
  echo "ERROR: A semicolon is missing."
  print_usage
  exit 1
fi >&2

# Take slices of the arguments that come before and after the semicolon.
declare -a lp2normal_command=("${args[@]:0:semicolon_index}")
declare -a clasp_command=("${args[@]:semicolon_index+1}")

# It is an error if either slice is empty.
for name in lp2normal clasp
do
  eval '
    if [[ "${#'"$name"'_command[@]}" = 0 ]]; then
      echo "ERROR: A path to '"$name"' is missing."
      print_usage
      exit 1
    fi
  '
done >&2

# It is an error if the first argument of either slice is not executable.
for name in lp2normal clasp
do
  eval '
    first="${'"$name"'_command[0]}"
    if ! [[ -e "$(which "$first")" ]]; then
      echo "ERROR: The \"'"$name"'\" path \"$first\" is not executable."
      print_usage
      exit 1
    fi
  '
done >&2

# Define the solving command.
if [[ "$normalize_only" = 0 ]]; then
  # Use the clasp command for solving.
  declare -a solve_command=("${clasp_command[@]}")
else
  # Skip solving.
  declare -a solve_command=(cat)
fi

#==============================================================================
# Set up a trap for some reason.
#

# Trap posix signals listed in "man 7 signal" that have "Action" "Term", except
# for user-defined signals.
trap "clean_end 1" SIGHUP SIGINT SIGKILL SIGPIPE SIGALRM SIGTERM

#==============================================================================
# Set up temporary files.
#

for name in input out err
do
  template="tmp-lp2normal-ous-clasp-$name.XXXXXXXXXX"
  eval '
    'tmp_"$name"'="$(mktemp --tmpdir="$TMP_DIR" "$template")"
  '
done

#==============================================================================
# Do important things.
#

# Grab standard input.
cat >"$tmp_input"

# Run clasp until a single model is found.
"${clasp_command[@]}" --models=1 "$tmp_input" >"$tmp_out" 2>"$tmp_err"
exit_status="$?"

# It is an error if clasp failed.
if [[ "$exit_status" = 1 ]]; then
  echo "ERROR: Initial clasp run failed with exit status $exit_status"
  if [[ -r "$tmp_err" ]]; then
    cat "$tmp_err"
  fi
  exit 1
fi >&2

# Look for an optimization value.
line="$(grep -E 'Optimization:[[:space:]]*+' "$tmp_out")"
value="$(grep -E --only-matching '[[:digit:]]+[[:space:]]*' <<<"$line")"

# It is an error if no optimization value was found.
if ! [[ "$value" =~ ^[[:digit:]]+$ ]]; then
  echo "ERROR: Initial clasp run resulted in no found optimization value"
  exit 1
fi >&2

if [[ "$normalize_only" = 0 ]]; then
  # Copy the optimization line to the output.
  echo "$line"
fi

# Normalize with the help of the optimization value and then run clasp again.
"${lp2normal_command[@]}" -ous"$value" "$tmp_input" | "${solve_command[@]}"
exit_status=$?

#==============================================================================
# Clean up.
#

clean_end "$exit_status"
