#!/bin/bash

# Translator for combining ASP and difference logic (DL)
# based on the approach described in the GTTV 2011 paper:
#
# T. Janhunen, G. Liu, and I. Niemela: Tight Integration of
# Non-Ground Answer Set Programming and Satisfiability Modulo Theories
#
# (c) 2011 Tomi Janhunen

DIR=.        # Consider replacing this by the absolute path (pwd)
TMP=$DIR/tmp
BIN=$DIR/bin
DEF=$DIR/def

# USAGE: dingo.sh [--ground-only] <gringo arguments>

# =============================================================================
# Local Functions

function cleanup()
{
  for f in $purge
  do
    rm -f $f
  done
}

function toclean()
{
  if [ $# -gt 0 ]
  then
    touch $*
    purge=$purge" "$*
    trap cleanup EXIT
  fi
}

# =============================================================================

ground=$TMP/$$-ground.sm
symfile=$TMP/$$-symbols.sm
tmpfile=$TMP/$$-file.s
theory=$TMP/$$-ground.dl
model=$TMP/$$-model.txt

# Declare temporary files for removal upon exit

toclean $ground $symfile $tmpfile $theory $model

# OPTIONS

if test "$1" = "--ground-only"
then
  let gnd=1
  shift
else
  let gnd=0
fi

# GROUND

$BIN/gringo $* $DEF/ext.lp \
| $BIN/lpcat -m -s=$symfile > $ground 2>> /dev/null
if test $? -ne 0
then
  echo "$0: grounding '""$*""' failed!"
  exit -1
fi

# EXTRACT SYMBOLS

cat $symfile \
| egrep -v -f $DEF/type.egrep > $tmpfile
rm -f $symfile
mv -f $tmpfile $symfile

# SMT-LIB (v. 1.2) HEADER

echo "(benchmark asp_plus_dl" > $theory
echo ":logic QF_IDL" >> $theory

# TYPE DEFINITIONS

egrep -f $DEF/type.egrep $ground \
| sed 's/[1-9][0-9]* //g' \
| sed 's/_/_u/g;s/(/_l/g;s/,/_c/g;s/)/_r/g' \
| sed 's/int_l/int(_/g;s/_r$/)/g' \
| cat $DEF/expand.m4 - \
| m4 - \
| egrep -v '^$' >> $theory
if test $? -ge 2
then
  echo "$0: failed to handle types of theory variables!"
  exit -1
fi

# ASP TRANSLATION

$BIN/lp2diff $ground | tail -n +3 | head -n -1 >> $theory

# THEORY ATOMS

read filter < $DEF/filter.egrep

egrep "$filter" $ground \
| $BIN/quote \
| sed 's/(/ /;s/)$//g' \
| awk '{printf("%s(%i,_%s,_%s,%s)",$2,$1,$3,$4,$5);}' \
| cat $DEF/expand.m4 - \
| m4 - \
| egrep -v '^$' \
| sed 's/(iff/  (iff/g' >> $theory
if test $? -ne 0
then
  echo "$0: failed to translate theory atoms themselves!"
  exit -1
fi

# SMT-LIB (v. 1.2) FOOTER

echo "))" >> $theory

if test $gnd -gt 0
then
  cat $theory
  exit 0
fi

# SOLVE

$BIN/z3 -smt -m $theory > $model
if test $? -ne 0
then
  echo "$0: z3 did not accept the input!"
  exit -1
fi

# PRINT

if egrep "^unsat$" $model 1>/dev/null
then
  echo "UNSATISFIABLE"
else
  if egrep "^sat$" $model 1>/dev/null
  then
    echo "SATISFIABLE"
    fgrep "var_" $model \
    | fgrep " -> true" \
    | sed 's/var_//g' \
    | sed 's/ -> true//g' \
    | $BIN/interpret $symfile \
    | egrep -v '^$'\
    | egrep -v '^'$filter'$' \
    | tr -d "\n"
    echo ""
    echo -n "Theory: "
    fgrep "var_" $model \
    | fgrep " -> true" \
    | sed 's/var_//g' \
    | sed 's/ -> true//g' \
    | $BIN/interpret $symfile \
    | egrep -v '^$'\
    | egrep '^'$filter'$' \
    | tr -d "\n"
    echo ""
    echo -n "Vars: "
    egrep '_[a-z].* -> [-]*[0-9][0-9]*' $model \
    | sed 's/_//;s/ -> /=/g' \
    | sed 's/_l/(/g;s/_c/,/g;s/_r/)/g;s/_u/_/g' \
    | tr "\n" " "
    echo ""
  else
    echo "UNKNOWN"
  fi
fi
