function sR=icassoStruct(X)
%function sR=icassoStruct([X])
%
%PURPOSE
%
%To initiate an Icasso result data structure which is meant for
%storing and keeping organized all data, parameters and results
%when performing the Icasso procedure. 
%
%EXAMPLE OF BASIC USAGE
%
%  S=icassoStruct(X);
%
%creates an Icasso result structure in workspace variable S. Its
%fields are initially empty except field .signal that contains
%matrix X. 
%
%INPUT 
%
%[An argument in brackets is optional. If it isn't  given or it's
% an empty matrix/string, the function will use a default value.] 
%
%[X] (dxN matrix) the original data (signals) consisting of N
%      d-dimensional vectors. Icasso centers the data (removes the 
%      sample mean from it) and stores it in field .signal. 
%      If the input argument is not given, or it is empty, 
%      the field is left empty.
%
%OUTPUT
%
% sR (struct) Icasso result structure that contains fields
%
%    .mode              (string)
%    .signal            (matrix)
%    .index             (matrix)
%    .fasticaoptions    (cell array)
%    .A                 (cell array)
%    .W                 (cell array)
%    .whiteningMatrix   (matrix)
%    .dewhiteningMatrix (matrix)
%    .cluster           (struct)
%    .projection        (struct)
%
%DETAILS
%
%The following table presents the fields of Icasso result
%structure. Icasso is a sequential procedure that is split into
%several phases (functions). The table shows the order in which 
%the fields are computed, the function that is used to change the
%parameters/results in the field, and lastly the other fields that
%the result depends on.  
%
%Phase Field                  Function               Depends on
%
%(1)  .mode                   icassoEst         (user input)   
%(1)  .signal                 icassoEst         (user input)
%(1)  .index                  icassoEst              
%(1)  .fasticaoptions         icassoEst         (user input)
%(1)  .A                      icassoEst              
%(1)  .W                      icassoEst              
%(1)  .whiteningMatrix        icassoEst              
%(1)  .dewhiteningMatrix      icassoEst              
%(2)  .cluster.simfcn         icassoSimilarity  (user input)
%(2)  .cluster.similarity     icassoSimilarity  (1)
%(3a) .cluster.strategy       icassoCluster     (user input)
%(3a) .cluster.partition      icassoCluster     (1-2)
%(3a) .cluster.dendrogram     icassoCluster     (1-2)
%(3a) .cluster.index.R        icassoCluster     (1-2)
%(3b) .projection.method      icassoProjection  (user input)
%(3b) .projection.parameters  icassoProjection  (user input)
%(3b) .projection.coordinates icassoProjection  (1-2)
%
%icassoEst performs phase 1.
%icassoExp performs phases 2-3 with default parameters.
%
%(1) Data, ICA parameters, and estimation results
%
%   .mode (string) 
%     type of randomization ('bootstrap'|'randinit'|'both')
%
%   .signal (dxN matrix) 
%     the original data (signal) X (centered) where N is 
%     the number of samples and d the dimension
%
%   .index  (Mx2 matrix) 
%     the left column is the number of the estimation cycle, the
%     right one is the number of the estimate on that cycle.
%     See also function: icassoGet
%
%The following fields contain parameters and results of the ICA
%estimation using FastICA toolbox. More information can be found,
%e.g., from of function fastica in FastICA toolbox.
%
%   .fasticaoptions (cell array) 
%     contains the options that FastICA uses in estimation.
%
%   .A (cell array of matrices) 
%     contains mixing matrices from each estimation cycle
%
%   .W (cell array of matrices) 
%     contains demixing matrices from each estimation cycle
%
%   .whiteningMatrix (matrix) 
%     whitening matrix for original data (sR.signal)
%
%   .dewhiteningMatrix (matrix) 
%     dewhitening matrix for original data (sR.signal).
%
%(2) Mutual similarities and clustering
%
%Parameters and results of 
% -computing similarities S between the estimates, and
% -clustering the estimates
%are stored in field .cluster which has the following subfields:
%
%   .cluster.simfcn (string)
%     the string option for function icassoSimilarity; it tells 
%     how mutual similarities between estimates are computed. 
%
%   .cluster.similarity (MxM matrix) 
%     mutual similarities between estimates.
%
%   .cluster.strategy (string) 
%     agglomeration strategy that was used for hierarchical
%     clustering which is done on dissimilarities D=1-S computed
%     from the similarity matrix S (in .cluster.similarity). 
%
%   .cluster.partition (MxM matrix) 
%     stores the partitions resulting clustering. Each row
%     partition(i,:), represents a division of M objects into K(i)
%     clusters (classes). On each row, clusters must be labeled
%     with integers 1,2,...,K(i), where K(i) is the number of
%     clusters that may be different on each row Example:
%     partition=[[1 2 3 4];[1 1 1 1];[1 1 2 2]] gives three
%     different partitions where partition(1,:) means every object
%     being in its own clusters; partition(2,:) means all objects
%     being in a single cluster, and in partition(3,:) objects 1&2
%     belong to cluster 1 and 3&4 to cluster 2.
%
%   .cluster.dendrogram.Z and .cluster.dendrogram.order
%     stores information needed for drawing dendrogram and
%     similarity matrix visualizations. More details in function
%     som_linkage 
%
%The following subfields of .cluster contain heuristic validity
%scores for the partitions in .cluster.partition. If the score is
%NaN it means that the validity has not been (or can't be)
%computed. 
%
%   .cluster.index.R (Mx1 vector) 
%     computed by subfunction rindex
%
%(3b) Projection for visualization
%
%Parameters for performing the visualization projection are results
%of the projection can be found in field .projection.
%
%  .projection has the following subfields:
%
%  .projection.method (string)
%     projection method used in icassoProjection
%
%  .projection.parameters (cell array)
%     contains parameters used in icassoProjection
%
%  .coordinates (Mx2 matrix)
%     contains the coordinates of the projected estimates
%

%COPYRIGHT NOTICE
%This function is a part of Icasso software library
%Copyright (C) 2003-2005 Johan Himberg
%
%This program is free software; you can redistribute it and/or
%modify it under the terms of the GNU General Public License
%as published by the Free Software Foundation; either version 2
%of the License, or any later version.
%
%This program is distributed in the hope that it will be useful,
%but WITHOUT ANY WARRANTY; without even the implied warranty of
%MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
%GNU General Public License for more details.
%
%You should have received a copy of the GNU General Public License
%along with this program; if not, write to the Free Software
%Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
%02111-1307, USA.

% ver 1.2 johan 100105

if nargin<1|isempty(X),
  X=[];
else
  X=remmean(X);
end

sR.mode=[]; 
sR.signal=X;
sR.index=[];
sR.fasticaoptions=[];
sR.A=[];
sR.W=[];
sR.whiteningMatrix=[];
sR.dewhiteningMatrix=[];
sR.cluster=initClusterStruct;
sR.projection=initProjectionStruct;

function cluster=initClusterStruct

cluster.simfcn=[];
cluster.similarity=[];
cluster.strategy=[];
cluster.partition=[];
cluster.dendrogram.Z=[];
cluster.dendrogram.order=[];
%cluster.index.balance=[];
cluster.index.R=[];

function projection=initProjectionStruct

projection.method=[];
projection.parameters=[];
projection.coordinates=[];
