function xy = itercov(varargin)
% ITERCOV - calculate covariance of a matrix with columns as
%           observations in several steps
%
% XY = itercov(X,...);
% 
% Mean value of every variable (row) will be removed before 
% calculating the covariance. 
%
% NOTE: cov(A',1) = itercov(A) + rounding errors
%
% xy = itercov(x)            - covariance matrix of x
% xy = itercov(x1, ..., xn)  - covariance matrix of [x1,...,xn]
%
% Copyright (C) Jaakko Väyrynen
  
  no = 0;                   % total number of samples
  xy = 0;                   % covariance matrix
  means = 0;                % means of variables
  args = length(varargin);  % number of argument matrices
  % Calculate mean of variables
  for a = 1 : args,
    [m, n] = size(varargin{a});
    if ~isempty(varargin{a}),
      means = means + sum(varargin{a}, 2);    
      no = no + n;
    else
      warning('Skipping empty data matrix.');
    end
  end
  means = means / no;

  % Calculate covariance matrix iteratively using at most 1000 samples
  % in one iteration. Simple multiplication takes too much memory
  % because of mean removal and looping sample by sample takes too
  % long.
  tic; 
  ndone = 0;                % number of processed samples
  lastt = +Inf;             % last estimated time left
  lastbtime = +Inf;         % time to complete last batch
  lastbsize = 400;
  bsize = 400;               % last batch size
  bstart = toc;             % time of batch start

  for a = 1 : length(varargin),
    if isempty(varargin{a}), continue; end
    [m,n] = size(varargin{a});  
    inds = 1:n; 
    while ~isempty(inds),
      tleft = fix(toc*(no/max(1,ndone)-1)); % eta
      if lastt>=tleft+60 & ndone > 1,    % print eta once a minute
	mins=fix(tleft/60);	
	secs=mod(tleft,60);
	if mins>0 | secs>0,
	  fprintf('itercov: Estimated time left [%d min %d sec]\n', ...
		  mins,secs);
	end
	lastt=tleft;
      end
      l = min(length(inds), bsize); % next # of samples
      ndone = ndone + l;    % we are doing them now
      k = inds(1:l);        % choose the indexed
      inds = inds(l+1:end);  
      xc = varargin{a}(:, k) - repmat(means,1,l); % remove mean
      xy = xy + xc*xc';     % add to covariance matrix
      
      if (toc-bstart)/bsize > lastbtime/lastbsize+0.001,
	%disp('Halfing batch size');
	lastbsize = bsize;
	bsize = max(1, fix(bsize/2));
      elseif (toc-bstart)/bsize < lastbtime/lastbsize-0.001 ...
	    & bsize < 2000,
	%disp('Doubling batch size');
	lastbsize = bsize;
	bsize = 2*bsize;
      end
      lastbtime = toc-bstart;       % time it took to complete the batch
      bstart = toc;             % time of next batch start
    end
  end

  % NOTE: changed on Fri 13th to normalize with 'no' instead of 'no-1'
  xy = xy / no; 

