/* -*- Mode: C; indent-tabs-mode: t; c-basic-offset: 2; tab-width: 2 -*-  */
/*
 * GeneralizedPropagator.h
 * Copyright (C) 2015 Shahab Tasharrofi <shahab@pc41>
 *
 * minisat-general-propagators is free software: you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the
 * Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * minisat-general-propagators is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License along
 * with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#ifndef _GENERALIZED_PROPAGATOR_H_
#define _GENERALIZED_PROPAGATOR_H_

#include <unordered_set>
#include <unordered_map>

#include "core/CallbackHandlerTypes.h"
#include "utils/FixedSizeIntSet.h"
#include "utils/FixedSizeIntMap.h"
#include "TriggerProtectedPropagator.h"

using namespace std;

class GeneralizedPropagator: public TriggerProtectedPropagator, public VarBumpActivityCallbackHandler
{
	private:
		bool triggersInitialized;
		bool hasCompleteEncoding;

		vec<Lit> reason;
		vec<Lit> assumptions;
		vec<Lit> explanationAssumptions;
		Solver *newSolverInstance;

		FixedSizeIntSet<EnumerabilityTrait> triggers; // set of literals that, if becoming true, trigger the propagator execution
		FixedSizeIntSet<EnumerabilityTrait> reasonLiterals; // set of literal that should be inserted into reason clause.
																												// keeping them as a set prevents multiple insertion of them.

		/* In the following, upperboundMap and lowerboundMap are the variables whose value
		 * will be assumed when search procedure of newSolverInstance is called.
		 * 
		 * Informally, (v --> v') \in upperboundMap means that our assumed value for v' would
		 * be the upperbound of current value for v (which could be unknown).
		 * 
		 * Similarly, (v --> v') \in lowerboundMap means that our assumed value for v' would
		 * be the lowerbound of current value for v (which could be unknown).
		 */
		FixedSizeIntMap<Var> upperboundMap; // mapping v --> v' sets v':=false if v==false and sets v':=true otherwise
		FixedSizeIntMap<Var> lowerboundMap; // mapping v --> v' sets v':=true if v==true and sets v':=false otherwise
		FixedSizeIntMap<Var> reverseMapping; // maps variables in upperbound or lowerbound to their source in external solver
																				 // i.e., if v --> v' in lower or upper bound then v' --> v in reverseMapping
																				 // used to translate a reasons and triggers into the language of external solver

		FixedSizeIntMap<int> varToAssumptionIndex;	// v --> i means that variable v is currently stored at position i of assumption vector
		int currentAssumptionIndex;									// Shows the last assumption index that is currently filled

		vector< pair<Lit, vector<Lit> > > reasonGenerator; // here l --> (l_1, ..., l_n) means that literal l should be 
																											 // included in the reason if all l_1 ... l_n are true

		bool performPropagations();
		void addAssumption(Lit p)
		{
			Var externalVar = var(p);

			if (upperboundMap.hasKey(externalVar))
			{
				Var internalVar = upperboundMap.valueOf(externalVar);

				int tmpNewPosition = varToAssumptionIndex.valueOf(internalVar);
				assumptions[tmpNewPosition] = assumptions[currentAssumptionIndex];
				assumptions[currentAssumptionIndex] = mkLit(internalVar, sign(p));
				varToAssumptionIndex.insert(internalVar, currentAssumptionIndex);
				varToAssumptionIndex.insert(var(assumptions[tmpNewPosition]), tmpNewPosition);
				currentAssumptionIndex++;
			}

			if (lowerboundMap.hasKey(externalVar))
			{
				Var internalVar = lowerboundMap.valueOf(externalVar);

				int tmpNewPosition = varToAssumptionIndex.valueOf(internalVar);
				assumptions[tmpNewPosition] = assumptions[currentAssumptionIndex];
				assumptions[currentAssumptionIndex] = mkLit(internalVar, sign(p));
				varToAssumptionIndex.insert(internalVar, currentAssumptionIndex);
				varToAssumptionIndex.insert(var(assumptions[tmpNewPosition]), tmpNewPosition);
				currentAssumptionIndex++;
			}
		}

		void removeAssumption(Lit p)
		{
			Var externalVar = var(p);

			if (upperboundMap.hasKey(externalVar))
			{
				Var internalUpVar = upperboundMap.valueOf(externalVar);
				int internalVarPos = varToAssumptionIndex.valueOf(internalUpVar);
				assumptions[internalVarPos] = mkLit(internalUpVar);
				currentAssumptionIndex--;
			}

			if (lowerboundMap.hasKey(externalVar))
			{
				Var internalLowVar = lowerboundMap.valueOf(externalVar);
				int internalVarPos = varToAssumptionIndex.valueOf(internalLowVar);
				assumptions[internalVarPos] = ~mkLit(internalLowVar);
				currentAssumptionIndex--;
			}
		}
	protected:
		// Overridden protected methods from Propagator class
		virtual bool propagate(int start, int end) override;
		virtual bool propagate(Lit p) override;
		virtual void cancelUntil(int level) override;
	public:
		GeneralizedPropagator(Solver *S, Solver *internalSolver, bool completeEncoding,
		                      int externalSolverVarCount, int internalSolverVarCount,
		                      const vector< pair<Var, Var> > &upperboundMapping,
		                      const vector< pair<Var, Var> > &lowerboundMapping,
		                      const vector< pair<Lit, vector<Lit> > > &conditionalReasons);

		// Overridden public methods from Propagator class
		virtual bool initialize(void) override;
		virtual const vec<Lit> &explain() override;
		virtual bool isTheoryVar(Var v) override;

		// Overridden public methods from TriggerProtectedPropagator class
		virtual bool triggersReady(void) override;
		virtual const vec<Lit> &getTriggers(void) override;

		// Overridden public methods from VarBumpActivityCallbackHandler class
		virtual void varBumpActivity(Var v) override;
};

#endif // _GENERALIZED_PROPAGATOR_H_

