/* -*- Mode: C; indent-tabs-mode: t; c-basic-offset: 2; tab-width: 2 -*-  */
/*
 * GeneralizedPropagator.cc
 * Copyright (C) 2015 Shahab Tasharrofi <shahab@pc41>
 *
 * minisat-general-propagators is free software: you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the
 * Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * minisat-general-propagators is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License along
 * with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include "GeneralizedPropagator.h"

#define noGEN_PROP_DEBUG

#ifdef GEN_PROP_DEBUG
#define noGEN_PROP_DEEP_DEBUG
#endif

#ifdef PROPAGATOR_DEBUG
#ifdef GEN_PROP_DEBUG
static const char *genBoolPropName = "bool";
#endif
#endif

bool GeneralizedPropagator::performPropagations()
{
#ifdef GEN_PROP_DEEP_DEBUG
	printf("generalized propagator: performing propagations\n");
#endif

	if (newSolverInstance->solve(assumptions))
	{
#ifdef GEN_PROP_DEBUG
		printf("generalized propagator (e: %d, i: %d): found model", SolverInstance->nClauses(), newSolverInstance->nClauses());
		for (int i = 0; i < newSolverInstance->model.size(); i++)
			printf(" "), newSolverInstance->printLiteral(stdout, mkLit(i, newSolverInstance->model[i] == l_False));
		printf(". generating a conflict\n");
#endif

		if (reasonGenerator.size() > 0)
		{ // If reason generator contains literals, then it means that we know how to
			// generate the reasons
			reason.clear();
			for (auto it = reasonGenerator.cbegin(); it != reasonGenerator.cend(); it++)
			{
				bool test = true;
				for (auto jt = (*it).second.cbegin(); jt != (*it).second.cend(); jt++)
					if (newSolverInstance->model[var(*jt)] != (sign(*jt) ? l_False : l_True))
					{
						test = false;
						break;
					}

				if (test)
					reason.push((*it).first);
			}
			Propagator::Effect e = addNewClause(reason, false);
			return (e != CONFLICT);
		}
		else
		{ // If not, the reason has to be generated from the clauses. The reason is
			// generated by taking one literal from each clause that is not satisfied
			// by internal variables of the solver. The literal should be such that it
			// satisfies the clause. The existence of such a literal is guaranteed by
			// the fact that we have found a model that satisfies all clauses.
			//
			// We then also add literals from the assumptions that explain the
			// satisfiability of solver's propagators. These are needed because the
			// solver might not have a complete axiomatization of the problem. This is
			// particularly true when QBF is going to be solved in a hierarchical
			// manner.
			auto findSatLiteral = [this](const Clause &C) -> Lit
			{
				Lit p = lit_Undef;
				for (int j = 0; j < C.size(); j++)
				{
					Lit l = C[j];
					if (reverseMapping.hasKey(var(l)))
					{
						Lit newLit = mkLit(reverseMapping.valueOf(var(l)), !sign(l));
						if (reasonLiterals.hasMember(newLit.x))
							return lit_Undef;
						else if (newSolverInstance->modelValue(l) == l_True)
							p = newLit;
					}
					else if (newSolverInstance->modelValue(l) == l_True)
						return lit_Undef;
				}
				assert(p != lit_Undef);
				return p;
			};

			reasonLiterals.clear();
			int undefLiteralCount = 0;
			Lit lastUndefLiteral = lit_Undef;

#ifdef GEN_PROP_DEEP_DEBUG
			printf("generalized propagator: generating conflict clause\n");
#endif
			for (int i = 0; i < newSolverInstance->getPropagatorCount(); i++)
			{
				const vec<Lit> &propReason = newSolverInstance->getPropagator(i)->explain();
				for (int j = 0; j < propReason.size(); j++)
				{
					Lit p = propReason[j];
					if (reverseMapping.hasKey(var(p)))
					{ // If the reason has to do with our assumptions, we have to add it to
						// the reasonLiterals
						assert(newSolverInstance->modelValue(p) == l_True);
						Lit newLit = mkLit(reverseMapping.valueOf(var(p)), !sign(p));

						assert(SolverInstance->value(newLit) != l_True);
						if (SolverInstance->value(newLit) == l_Undef)
						{
							undefLiteralCount++;
							lastUndefLiteral = newLit;
						}

#ifdef GEN_PROP_DEEP_DEBUG
						printf("generalized propagator: adding literal ");
						SolverInstance->printLiteral(stdout, newLit);
						printf(" because of propagator explanation\n");
#endif
						reasonLiterals.insert(newLit.x);
					}
				}
			}

			for (int i = 0; i < newSolverInstance->nBoundVars(); i++)
			{
				Lit p = newSolverInstance->getBoundLitAt(i);
				if (reverseMapping.hasKey(var(p)))
				{ // If literal p is forced in the internal solver, it has to be present
					// in all reasons (because it is part of all hitting sets)
					assert(newSolverInstance->modelValue(p) == l_True);
					Lit newLit = mkLit(reverseMapping.valueOf(var(p)), !sign(p));

					assert(SolverInstance->value(newLit) != l_True);
					if (SolverInstance->value(newLit) == l_Undef)
					{
						undefLiteralCount++;
						lastUndefLiteral = newLit;
					}

#ifdef GEN_PROP_DEEP_DEBUG
					printf("generalized propagator: adding literal ");
					SolverInstance->printLiteral(stdout, newLit);
					printf(" because it is forced by the internal solver\n");
#endif
					reasonLiterals.insert(newLit.x);
				}
			}
#ifdef GEN_PROP_DEEP_DEBUG
			printf("generalized propagator: checking %d clauses from internal solver\n", newSolverInstance->nClauses());
#endif
			for (int i = 0; i < newSolverInstance->nClauses(); i++)
			{
				const Clause &c = newSolverInstance->getClause(i);
				Lit newLit = findSatLiteral(c);

				if (newLit != lit_Undef)
				{
					assert(SolverInstance->value(newLit) != l_True);
					if (SolverInstance->value(newLit) == l_Undef)
					{
						undefLiteralCount++;
						lastUndefLiteral = newLit;
					}

					reasonLiterals.insert(newLit.x);

#ifdef GEN_PROP_DEEP_DEBUG
					printf("generalized propagator: literal ");
					newSolverInstance->printLiteral(stdout, newLit);
					printf(" chosen because of clause ");
					for (int counter = 0; counter < c.size(); counter++)
					{
						newSolverInstance->printLiteral(stdout, c[counter]);
						printf(" ");
					}
					printf("\n");
#endif
				}
#ifdef GEN_PROP_DEEP_DEBUG
				else
				{
					printf("generalized propagator: no literal chosen from clause ");
					for (int counter = 0; counter < c.size(); counter++)
					{
						newSolverInstance->printLiteral(stdout, c[counter]);
						printf(" ");
					}
					printf("\n");
				}
#endif
			}

			assert(undefLiteralCount == 0);
			reason.clearunboxed();
			for (int i = reasonLiterals.begin(); i != reasonLiterals.end(); i = reasonLiterals.nextNumber(i))
				reason.push(mkLit(i >> 1, i & 1));
			Propagator::Effect e = addNewClause(reason, false);
			return (e != CONFLICT);
		}
	}
	else
	{
#ifdef GEN_PROP_DEEP_DEBUG
		printf("generalized propagator (e: %d, i: %d): no model found. finding triggers\n", SolverInstance->nClauses(), newSolverInstance->nClauses());
#endif

		triggers.clear();
		for (int i = 0; i < newSolverInstance->conflict.size(); i++)
		{
			Lit p = newSolverInstance->conflict[i];
			assert(reverseMapping.hasKey(var(p)));
			Var origVar = reverseMapping.valueOf(var(p));

			// Upper bound variables only appear in negative places of the internal
			// formula and thus a negative literal p is related to an upperbound
			// variable and should thus be watched in negative.
			//
			// Similarly, lower bound variables only appear in positive places of
			// the internal formula and thus positive literal p should be watched
			// positively.
			//
			// In both cases, a literal should only be watched if its value differs
			// from its sign. That is positive variables should only be watched if
			// their value is not true and negative variables should only be watched
			// if their value is not false.

			Lit newLit = mkLit(origVar, sign(p));
			assert(SolverInstance->value(newLit) != l_True); // because it is part of the conflict
			triggers.insert(newLit.x);
		}
		triggersInitialized = true;
	}

	return true;
}

GeneralizedPropagator::GeneralizedPropagator(Solver *S, Solver *internalSolver, bool completeEncoding,
																						 int externalSolverVarCount, int internalSolverVarCount,
                                             const vector< pair<int, int> > &upperboundMapping,
                                             const vector< pair<int, int> > &lowerboundMapping,
                                             const vector< pair<Lit, vector<Lit> > > &conditionalReasons)
	: TriggerProtectedPropagator(S),
		triggers(externalSolverVarCount * 2 + 3),
		reasonLiterals(externalSolverVarCount * 2 + 3),
		upperboundMap(externalSolverVarCount),
		lowerboundMap(externalSolverVarCount),
		reverseMapping(internalSolverVarCount),
		varToAssumptionIndex(internalSolverVarCount)
{
#ifdef PROPAGATOR_DEBUG
#ifdef GEN_PROP_DEBUG
	debuggingEnabled = true;
	propagatorName = genBoolPropName;
#endif
#endif
	newSolverInstance = internalSolver;
	hasCompleteEncoding = completeEncoding;

	assumptions.clear();
	currentAssumptionIndex = 0;
	upperboundMap.clear();
	for (auto it = upperboundMapping.cbegin(); it != upperboundMapping.cend(); it++)
	{
		upperboundMap.insert((*it).first, (*it).second);
		reverseMapping.insert((*it).second, (*it).first);
		varToAssumptionIndex.insert((*it).second, assumptions.size());
		assumptions.push(mkLit((*it).second));
	}

	lowerboundMap.clear();
	for (auto it = lowerboundMapping.cbegin(); it != lowerboundMapping.cend(); it++)
	{
		lowerboundMap.insert((*it).first, (*it).second);
		reverseMapping.insert((*it).second, (*it).first);
		varToAssumptionIndex.insert((*it).second, assumptions.size());
		assumptions.push(~mkLit((*it).second));
	}

	reasonGenerator.clear();
	for (size_t i = 0; i < conditionalReasons.size(); i++)
	{
		reasonGenerator.push_back(pair<Lit, vector<Lit> >(conditionalReasons[i].first, vector<Lit>()));
		reasonGenerator[i].first = conditionalReasons[i].first;
		for (size_t j = 0; j < conditionalReasons[i].second.size(); j++)
			reasonGenerator[i].second.push_back(conditionalReasons[i].second[j]);
	}

	triggers.clear();
	triggersInitialized = false;

	// register itself as the var bump activity handler for the internal solver
	if (newSolverInstance->isVarBumpPropagationEnabled())
		newSolverInstance->setVarBumpCallbackHandler(this);
}

bool GeneralizedPropagator::initialize(void)
{
	return true;
}

bool GeneralizedPropagator::propagate(int start, int end)
{
#ifdef GEN_PROP_DEEP_DEBUG
	printf("generalized propagator (e: %d, i: %d): propagating literals ", SolverInstance->nClauses(), newSolverInstance->nClauses());
	for (int i = start; i < end; i++)
		printf("%d ", ConvertLiteralToInt(getTrailLiteralAt(i)));
	printf("\n");
#endif

	for (int i = start; i < end; i++)
		addAssumption(getTrailLiteralAt(i));

	if (!triggersInitialized)
		return !performPropagations();
	else
	{
		bool test = false;
		for (int i = start; i < end; i++)
			if (triggers.hasMember(getTrailLiteralAt(i).x))
			{
				test = true;
				break;
			}

		if (test)
			return !performPropagations();
		else
		{
#ifdef GEN_PROP_DEEP_DEBUG
				printf("generalized propagator: performing propagation method was not necessary\n");
#endif
				return false;
		}
	}
}

bool GeneralizedPropagator::propagate(Lit p)
{
#ifdef GEN_PROP_DEEP_DEBUG
	printf("generalized propagator: propagating literal %d\n", ConvertLiteralToInt(p));
#endif
	addAssumption(p);
	if (!triggersInitialized || (triggers.hasMember(p.x)))
		return !performPropagations();
	else
	{
#ifdef GEN_PROP_DEEP_DEBUG
		printf("generalized propagator: performing propagation method was not necessary\n");
#endif
		return false;
	}
}

void GeneralizedPropagator::cancelUntil(int level)
{
	for (int i = propagationStartIndex - 1; i >= getTrailLimit(level); i--)
		removeAssumption(getTrailLiteralAt(i));
	triggersInitialized = false;
}

bool GeneralizedPropagator::isTheoryVar(Var v)
{
	return upperboundMap.hasKey(v) || lowerboundMap.hasKey(v);
}

const vec<Lit> &GeneralizedPropagator::explain()
{
#ifdef GEN_PROP_DEBUG
	printf("generalized propagator: explaining current model\n");
#endif
	// Initialize assumptions based on the current model found by the external solver
	explanationAssumptions.clear();
	for (int i = 0; i < SolverInstance->model.size(); i++)
	{
		if (upperboundMap.hasKey(i))
			explanationAssumptions.push(mkLit(upperboundMap.valueOf(i), SolverInstance->model[i] == l_False));
		if (lowerboundMap.hasKey(i))
			explanationAssumptions.push(mkLit(lowerboundMap.valueOf(i), SolverInstance->model[i] != l_True));
	}

	// Solve internal solver according to those assumptions. It should be UNSAT
	// because, if not, some conflict should have been returned previously and a
	// model should not have been found.
	if (newSolverInstance->solve(explanationAssumptions))
		assert(false); // Why is it satisfied? Did hell just break loose?

	// Now, explain satisfiability of this propagator according to the conflict
	// received from the internal solver
	reason.clear();
	for (int i = 0; i < newSolverInstance->conflict.size(); i++)
	{
		Lit p = newSolverInstance->conflict[i];
		assert(reverseMapping.hasKey(var(p)));
		Var origVar = reverseMapping.valueOf(var(p));

		Lit newLit = mkLit(origVar, !sign(p));
		assert(SolverInstance->modelValue(newLit) == l_True); // because it is the model value for the negation of a conflict literal 
		reason.push(newLit);
	}

#ifdef GEN_PROP_DEBUG
	printf("generalized propagator: explanation is ");
	for (int i = 0; i < reason.size(); i++)
		printf("%d ", ConvertLiteralToInt(reason[i]));
	printf("\n");
#endif

	return reason;
}

bool GeneralizedPropagator::triggersReady(void)
{
	return triggersInitialized;
}

const vec<Lit> &GeneralizedPropagator::getTriggers(void)
{
	reason.clearunboxed();
	for (int i = triggers.begin(); i != triggers.end(); i = triggers.nextNumber(i))
		reason.push(mkLit(i >> 1, i & 1));

	return reason;
}

void GeneralizedPropagator::varBumpActivity(Var v)
{
	if (reverseMapping.hasKey(v))
		SolverInstance->publishedVarBumpActivity(reverseMapping.valueOf(v));
}

