/* -*- Mode: C; indent-tabs-mode: t; c-basic-offset: 2; tab-width: 2 -*-  */
/*
 * BDDPropagator.cc
 * Copyright (C) 2015 Shahab Tasharrofi <>
 *
 * graphsat is free software: you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the
 * Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * graphsat is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License along
 * with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include "BDDPropagator.h"

#define noBDD_PROPAGATOR_DEBUG

#ifdef BDD_PROPAGATOR_DEBUG
#define noDEEP_BDD_PROPAGATOR_DEBUG
#endif

BDDPropagator::BDDPropagator(TriggerProtectedPropagator *P)
	: Propagator(P->getSolver()),
		bddManager(2 * P->getSolver()->nVars() + 1)
{
	internalPropagator = P;
	satisfiabilityLowerbound = bddManager.bddZero();
}

bool BDDPropagator::initialize(void)
{
	return internalPropagator->initialize();
}

bool BDDPropagator::isTriggered()
{
	DdNode *currentNode = satisfiabilityLowerbound.getNode();
	bool negated = false;
	while (!Cudd_IsConstant(currentNode))
	{
		if (currentNode != Cudd_Regular(currentNode))
		{
			negated = !negated;
			currentNode = Cudd_Regular(currentNode);
		}
		Lit p  = { currentNode->index - 1 };
		if (SolverInstance->value(p) == l_True)
			currentNode = Cudd_T(currentNode);
		else
			currentNode = Cudd_E(currentNode);
	}

	return (currentNode == (negated ? bddManager.bddOne().getNode() : bddManager.bddZero().getNode()));
}

void BDDPropagator::strengthenTriggers()
{
	if (!internalPropagator->triggersReady())
	{
#ifdef BDD_PROPAGATOR_DEBUG
		printf("BDD propagator wrapper: Skipping trigger strengthening because internal propagator has not computed triggers.");
#endif
		return;
	}

#ifdef BDD_PROPAGATOR_DEBUG
	printf("BDD propagator wrapper: strengthening triggers with cube ~V{ ");
#endif
	BDD curTrigger = bddManager.bddZero();
	const vec<Lit> &triggers = internalPropagator->getTriggers();
	for (int i = 0; i < triggers.size(); i++)
	{
		curTrigger += bddManager.bddVar(triggers[i].x + 1);
#ifdef BDD_PROPAGATOR_DEBUG
		SolverInstance->printLiteral(stdout, triggers[i]);
		printf(" ");
#endif
	}
#ifdef BDD_PROPAGATOR_DEBUG
	printf("}\n");
	printf("BDD propagator wrapper: BDD rep. for new sat condition is %s\n", (!curTrigger).FactoredFormString().c_str());
#endif
		
	satisfiabilityLowerbound += !curTrigger;
#ifdef DEEP_BDD_PROPAGATOR_DEBUG
	printf("BDD propagator wrapper: Satisfiability lowerbound is %s\n", satisfiabilityLowerbound.FactoredFormString().c_str());
#endif
}

bool BDDPropagator::propagate(int start, int end)
{
#ifdef BDD_PROPAGATOR_DEBUG
	printf("BDD propagator wrapper: propagating trail literals { ");
	for (int i = start; i < end; i++)
	{
		SolverInstance->printLiteral(stdout, getTrailLiteralAt(i));
		printf(" ");
	}
	printf("}\n");
#endif
	if (!isTriggered())
	{
#ifdef BDD_PROPAGATOR_DEBUG
		printf("BDD propagator wrapper: propagation not needed\n");
#endif
		return false;
	}

#ifdef BDD_PROPAGATOR_DEBUG
	printf("BDD propagator wrapper: running internal propagator\n");
#endif
	bool result = internalPropagator->propagateAll();
	if (result)
	{
#ifdef BDD_PROPAGATOR_DEBUG
		printf("BDD propagator wrapper: conflict found\n");
#endif
		setConflictClause(internalPropagator->getConflictClause());
		return true;
	}
	else
	{
		strengthenTriggers();
		return false;
	}
}

bool BDDPropagator::propagate(Lit p)
{
#ifdef BDD_PROPAGATOR_DEBUG
	printf("BDD propagator wrapper: propagating literal ");
	SolverInstance->printLiteral(stdout, p);
	printf("\n");
#endif
	if (!isTriggered())
	{
#ifdef BDD_PROPAGATOR_DEBUG
		printf("BDD propagator wrapper: propagation not needed\n");
#endif
		return false;
	}

#ifdef BDD_PROPAGATOR_DEBUG
	printf("BDD propagator wrapper: running internal propagator\n");
#endif
	bool result = internalPropagator->propagateAll();
	if (result)
	{
#ifdef BDD_PROPAGATOR_DEBUG
		printf("BDD propagator wrapper: conflict found\n");
#endif
		return true;
	}
	else
	{
#ifdef BDD_PROPAGATOR_DEBUG
		printf("BDD propagator wrapper: strengthening triggers\n");
#endif
		strengthenTriggers();
		return false;
	}
}

const vec<Lit> &BDDPropagator::explain()
{
#ifdef BDD_PROPAGATOR_DEBUG
	printf("BDD propagator wrapper: explaining satisfiability\n");
#endif
	explanation.clearunboxed();

	DdNode *currentNode = satisfiabilityLowerbound.getNode();
	bool negated = false;
	while (!Cudd_IsConstant(currentNode))
	{
		if (currentNode != Cudd_Regular(currentNode))
		{
			negated = !negated;
			currentNode = Cudd_Regular(currentNode);
		}
		Lit p  = { currentNode->index - 1 };
		if (SolverInstance->modelValue(p) == l_True)
			currentNode = Cudd_T(currentNode);
		else
		{
			explanation.push(~p);
			currentNode = Cudd_E(currentNode);
		}
	}

	if (currentNode == (negated ? bddManager.bddZero().getNode() : bddManager.bddOne().getNode()))
	{
#ifdef BDD_PROPAGATOR_DEBUG
		printf("BDD propagator wrapper: explanation is ");
		for (int i = 0; i < explanation.size(); i++)
		{
			SolverInstance->printLiteral(stdout, explanation[i]);
			printf(" ");
		}
		printf("\n");
#endif
		return explanation;
	}
	else
		return internalPropagator->explain();
}

void BDDPropagator::cancelUntil(int level)
{
	internalPropagator->backjump(level);
}

bool BDDPropagator::isTheoryVar(Var v)
{
	return internalPropagator->isTheoryVar(v);
}

