/* -*- Mode: C; indent-tabs-mode: t; c-basic-offset: 2; tab-width: 2 -*-  */
/*
 * prefix-tree.h
 * Copyright (C) 2016 Shahab <shahab@tasharrofi.net>
 *
 * grounder-generator is free software: you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the
 * Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * grounder-generator is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License along
 * with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#ifndef _PREFIX_TREE_H_
#define _PREFIX_TREE_H_

#include <boost/functional/hash.hpp>
#include <limits>
#include <stack>
#include <tuple>
#include <unordered_map>
#include <vector>

using namespace std;


template <typename T>
class PrefixTree
{
	public:
		typedef size_t PrefixIndexType;
		static const PrefixIndexType nullPrefix = numeric_limits<PrefixIndexType>::max();

		typedef pair<PrefixIndexType, T> PrefixDataType;

		struct PrefixDataHash
		{
			public:
				inline size_t operator()(const PrefixDataType &data) const { return boost::hash_value<PrefixDataType>(data); }
		};

	private:
		static vector<PrefixDataType> prefixes;
		static unordered_map<PrefixDataType, PrefixIndexType, PrefixDataHash> forwardIndex;
		static stack<T> operativeStack;

		inline static bool isEmpty(PrefixIndexType index) { return (index == nullPrefix); }
		inline static const T &getLastElement(PrefixIndexType index) { assert(index < prefixes.size()); return prefixes[index].second; }
		inline static PrefixIndexType getPrefix(PrefixIndexType index) { assert(index < prefixes.size()); return prefixes[index].first; }

		static PrefixIndexType append(PrefixIndexType s, const T &e)
		{
			PrefixDataType data(s, e);
			auto it = forwardIndex.find(data);
			if (it != forwardIndex.end())
				return it->second;

			PrefixIndexType lastIndex = prefixes.size();
			prefixes.push_back(data);
			forwardIndex[data] = lastIndex;

			return lastIndex;
		}

		static PrefixIndexType concat(PrefixIndexType s, PrefixIndexType t)
		{
			size_t lastIndex = operativeStack.size();
			while (!isEmpty(t))
			{
				operativeStack.push(getLastElement(t));
				t = getPrefix(t);
			}

			while (operativeStack.size() > lastIndex)
			{
				s = append(s, operativeStack.top());
				operativeStack.pop();
			}

			return s;
		}

		static void printToStream(ostream &os, PrefixIndexType t)
		{
			size_t lastIndex = operativeStack.size();
			while (!isEmpty(t))
			{
				operativeStack.push(getLastElement(t));
				t = getPrefix(t);
			}

			while (operativeStack.size() > lastIndex)
			{
				os << operativeStack.top();
				operativeStack.pop();
			}
		}
	public:
		class Sequence
		{
			private:
				PrefixIndexType index;

				Sequence(PrefixIndexType i) : index(i) { }
			public:
				inline bool isEmpty() const { return PrefixTree::isEmpty(index); }

				inline const T &getLastElement() const { return PrefixTree::getLastElement(index); }
				inline Sequence getPrefix() const { return Sequence(PrefixTree::getPrefix(index)); }

				inline Sequence operator+=(const Sequence &s) { index = PrefixTree::concat(index, s.index); return (*this); }
				inline Sequence operator+=(const T &element) { index = PrefixTree::append(index, element); return (*this); }

				inline bool operator==(const Sequence &s) const { return index == s.index; }
				inline bool operator!=(const Sequence &s) const { return index != s.index; }

				inline size_t getHashValue() const { return hash<PrefixIndexType>()(index); }
				inline void printToStream(ostream &os) const { PrefixTree::printToStream(os, index); }

				Sequence() : index(nullPrefix) { }
		};
};

template <typename T>
vector<typename PrefixTree<T>::PrefixDataType> PrefixTree<T>::prefixes;

template <typename T>
unordered_map<typename PrefixTree<T>::PrefixDataType, typename PrefixTree<T>::PrefixIndexType, typename PrefixTree<T>::PrefixDataHash> PrefixTree<T>::forwardIndex;

template <typename T>
stack<T> PrefixTree<T>::operativeStack;


template <typename T>
inline typename PrefixTree<T>::Sequence operator+(const typename PrefixTree<T>::Sequence &s, const typename PrefixTree<T>::Sequence &t)
{
	typename PrefixTree<T>::Sequence result(s);
	result += t;
	return result;
}

template <typename T>
inline typename PrefixTree<T>::Sequence operator+(const typename PrefixTree<T>::Sequence &s, const T &e)
{
	typename PrefixTree<T>::Sequence result(s);
	result += e;
	return result;
}

#endif // _PREFIX_TREE_H_

