/* -*- Mode: C; indent-tabs-mode: t; c-basic-offset: 2; tab-width: 2 -*-  */
/*
 * assignment-manager.h
 * Copyright (C) 2016 Shahab <shahab@tasharrofi.net>
 *
 * grounder-generator is free software: you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the
 * Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * grounder-generator is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License along
 * with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#ifndef _ASSIGNMENT_MANAGER_H_
#define _ASSIGNMENT_MANAGER_H_

#include <boost/variant.hpp>
#include <stack>
#include <unordered_map>
#include <vector>

using namespace std;

typedef int64_t VariableType;
typedef int64_t ValueType;

class AssignmentManager
{
	public:
		class AssignmentIterator;
	private:
		typedef size_t IndexType;
		static IndexType undefIndex;

		struct ClearAction
		{
			IndexType previousIteratorStartPosition;
		};

		struct AssignmentAction
		{
			VariableType variable;
			ValueType value;
			IndexType previousIndex;
		};

		typedef boost::variant<ClearAction, AssignmentAction> Action;

		// "lastAssignment" is used for fast random access to assigned value of a variable
		// if variable is not in lastAssignment, it means that, since the last clear action,
		// the variable has not been assigned yet.
		// The value of variable var can be obtained as follows:
		//   lastAssignment[var].getValue()
		// This works because we guarantee that the AssignmentIterator always points to the
		// right position in iteratableAssignments vector
		static unordered_map<VariableType, AssignmentIterator> lastAssignment;
		// "iteratableAssignments" is used for stateless iteration over current assignments.
		// The set of pairs of variables contained in the iteratableAssignments vector
		// after iteratorStartPosition are always in one-to-one correspondence with variables
		// in "lastAssignment" mapping. That is, lastAssignment[var]=index implies that
		// iteratableAssignments[index].first = var and index >= iteratorStartPosition. Also,
		// if index >= iteratorStartPosition then iteratableAssignments[index].first = var
		// imples that lastAssignment[var] = index.
		//
		// The difference between native iterators of lastAssignment map and iterators based
		// on iteratableAssignments vector is that the former iterators get invalidated if
		// lastAssignment is changed after obtaining those iterators. However, the latter
		// iterators will remain valid as long as changes to lastAssignment are undone before
		// accessing that iterator again.
		static vector< pair<VariableType, IndexType> > iteratableAssignments;
		// "actions" is a sequence of assignment actions that are applied one after another.
		// Each Action element saves information necessary for undoing this last action.
		//   1 - AssignmentAction saves the variable that is going to be reversed plus what
		//       position it was previously assigned in. Using "undefIndex" for position
		//       means that this variable has been previously unassigned, and, hence, when
		//       undoing, it should be removed from "lastAssignment" and popped back from
		//       "iteratableAssignments".
		//   2 - ClearAction saves the previous iterator start position so that, when
		//       undoing, the previously available variables in "lastAssignment" can be
		//       restored.
		static vector<Action> actions;
		// "undoPoints" marks the position in the action sequence that we should refer back
		// to when undoing. This way, we can undo several actions together.
		static stack<IndexType> undoPoints;
		// "iteratorStartPosition" defines the starting position in the iteratableAssignments
		// vector in order to iterate over currently assigned variables (which all come after
		// this position).
		static IndexType iteratorStartPosition;

		static void undoAssignment(const AssignmentAction &a);
		static void undoClear(const ClearAction &c);

		inline static void undoLastAction(const AssignmentAction &a) { undoAssignment(a); }
		inline static void undoLastAction(const ClearAction &c) { undoClear(c); }
	public:
		class AssignmentIterator
		{
			private:
				size_t currentIndex;

				AssignmentIterator(size_t index) : currentIndex(index) { }
			public:
				inline bool atEnd() const { return (currentIndex >= iteratableAssignments.size()); }
				inline VariableType getVariable() const { assert(!atEnd()); return iteratableAssignments[currentIndex].first; }
				inline ValueType getValue() const { assert(!atEnd()); return get<AssignmentAction>(actions[iteratableAssignments[currentIndex].second]).value; }

				inline AssignmentIterator operator++() { assert(!atEnd()); currentIndex++; return (*this); }
				inline AssignmentIterator operator++(int) { AssignmentIterator result = *this; ++(*this); return result; }
				AssignmentIterator() : currentIndex(iteratorStartPosition) { }

				friend class AssignmentManager;
		};

		inline static void newUndoPoint() { undoPoints.push(actions.size()); }
		static void pushAssignment(VariableType variable, ValueType value);
		static void clear();
		static void undo();

		inline static ValueType getValueOf(VariableType variable)
		{
			auto it = lastAssignment.find(variable);
			if (it == lastAssignment.end())
				return 0;
			else
				return it->second.getValue();
		}
};

typedef AssignmentManager assignments;

#endif // _ASSIGNMENT_MANAGER_H_

