function ri=rindex(dist,partition)
%function ri=rindex(dist,partition)
%
%PURPOSE
%
%To compute a clustering validity index called R-index to assist in
%selecting a proper number of clusters. Input argument
%'partition' defines different clustering results, partitions, of
%the same data. Clustering result that has minimum value for
%R-index among the compared ones is the "best" one.  
%
%The index is not computed (NaN) if the partition contains a
%cluster consisting of only one item
%
%INPUT
%
% dist      (NxN matrix) matrix of pairwise dissimilarities between N objects
% partition (KxN matrix) K partition vectors of the same N objects (see
%            explanation for 'partition vector' in function hcluster) 
%
%OUTPUT
%
% ri (Kx1 vector) ri(k) is the value of R-index for clustering
%      given in partition(k,:).
%
%DETAILS
%
%R-index is a Davies-Bouldin type relative clustering validity
%index. The clustering that has the minimum value among is the best
%in terms of R-index. The index is originally defined in Levine &
%Dormay (2001),  "Resampling method for unsupervised estimation of
%cluster validity. Neural Computation, and redescribed in
%publication Himberg et al. (2004), "Validating the independent
%components of neuroimaging time-series via clustering and
%visualization". Neuroimage, 22:3(1214-1222). 13(11)
%
%Note: if the partition contains clusters consisting of only one
%item, the index is especially doubtful and value NaN is assigned
%to the solution.
%
%SEE ALSO
% icassoRindex
% hcluster
% icassoStruct

%COPYRIGHT NOTICE
%This function is a part of Icasso software library
%Copyright (C) 2003-2005 Johan Himberg
%
%This program is free software; you can redistribute it and/or
%modify it under the terms of the GNU General Public License
%as published by the Free Software Foundation; either version 2
%of the License, or any later version.
%
%This program is distributed in the hope that it will be useful,
%but WITHOUT ANY WARRANTY; without even the implied warranty of
%MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
%GNU General Public License for more details.
%
%You should have received a copy of the GNU General Public License
%along with this program; if not, write to the Free Software
%Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

% ver 1.2 100105 johan

N=size(partition,1);

% Number of clusters in each partition
Ncluster=max(partition');

% This is for computing the waitbar
completed=cumsum(Ncluster.^2);

for k=1:N,
  if any(hist(partition(k,:),1:Ncluster(k))==1),
    % contains one-item clusters (index very doubtful)
    ri(k,1)=NaN;
  elseif Ncluster(k)==1,
    % Degenerate partition (all in the same cluster)
    ri(k,1)=NaN;
  else
    % Initialize internal scatter vector and between scatter matrix 
    Sin=zeros(Ncluster(k),1); 
    Sout=zeros(Ncluster(k));
    for i=1:Ncluster(k),
      % Find inter-cluster distances and compute their average
      thisPartition=find(i==partition(k,:));
      d_=dist(thisPartition,thisPartition); 
      Sin(i,1)=mean(d_(:));
      
      % Compute cluster min distances
      for j=i+1:Ncluster(k),
        d_=dist(thisPartition,j==partition(k,:)); 
        Sout(i,j)=min(d_(:));
      end 
    end
    % Make between-cluster distance matrix symmetric, self-distance
    % to Inf (we take min)
    Sout=Sout+Sout'; Sout(eye(Ncluster(k))==1)=Inf;
    % Compute R index
    ri(k,1)=mean(Sin(:)'./min(Sout)); 
  end
  clc;
  disp(sprintf('Computing R-index: %D % completed', round(100*completed(k)/completed(end)))),
end

