function sR=icassoCluster(sR,strategy,L)
%function sR=icassoCluster(sR,[strategy],L)
%
%PURPOSE 
%
%To cluster the ICA estimates according to the readily computed
%similarity matrix S in sR.cluster.dissimilarity. Uses hierarchical
%clustering on distance matrix. The distances are computed simply
%by D=1-S. 
%
%EXAMPLE OF BASIC USAGE
%
% sR=icassoCluster(sR); 
% 
%where sR is an Icasso result structure. This applies hierarchical
%clustering using group-average linkage agglomeration strategy and
%stores the results back into workspace variable sR.
%
%INPUT 
%
%[An argument in brackets is optional. If it isn't  given or it's
% an empty matrix/string, the function will use a default value.] 
%
% sR         (struct) Icasso result struct
% [strategy] (string) 'AL' (default) | 'SL' | 'CL' 
%             agglomeration strategy, (linkage) 
%              'AL' group average linkage
%              'SL' single linkage (nearest neighbor)
%              'CL' complete linkage (furthest neighbor)
% [L]        (string) 'auto' (default) or integer
%             computes a relative cluster validity index (function
%             rindex) for 2...L clusters. Default 'auto' sets a
%             reasonable L automatically (see details)
%
%OUTPUT
%
% sR (struct) updated Icasso result data structure 
%
%The function updates _only_ the following fields:
%sR.cluster.strategy, sR.cluster.similarity, sR.cluster.dendrogram,
%and sR.cluster.index.R. See icassoStruct.
%
%DETAILS
%
%1. Applies hierarchical agglomerative clustering on D
%   (D=1-sR.cluster.similarity), i.e., the dissimilarity matrix
%   using the selected strategy.
%2. Computes matrix P (of size MxM) where each row is a
%   partition. See function hcluster. The partitions present the
%   clustering of the objects on each level of the dendrogram. Let
%   c=P(L,i); Now, c is the cluster that object i belongs to at
%   level L. On each row P(L,:), cluster labels are integers
%   1,2,...,L. Stores P in field sR.cluster.partition (Stores also
%   outputs Z and order from function hcluster into
%   sR.cluster.dendrogram) 
%3. Computes a relative clustering validity index for partitions up
%   to L : determines L reasonably L=ceil(3*original data dimension)
%   however, L is no more than there are estimates, 
%   and in maximum 100 (too slow for bigger L). Stores the result
%   in field sR.cluster.index.R   
%
%Note: the cluster labels cannot be compared over partitions. The
%same cluster may appear at different level(s) but it does not
%necessarily have the same label in every 'partition vector' P(L,:). 
%
%SEE ALSO
% hcluster
% som_linkage
% rindex
% icassoExp
% icassoDendrogram

%COPYRIGHT NOTICE
%This function is a part of Icasso software library
%Copyright (C) 2003-2005 Johan Himberg
%
%This program is free software; you can redistribute it and/or
%modify it under the terms of the GNU General Public License
%as published by the Free Software Foundation; either version 2
%of the License, or any later version.
%
%This program is distributed in the hope that it will be useful,
%but WITHOUT ANY WARRANTY; without even the implied warranty of
%MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
%GNU General Public License for more details.
%
%You should have received a copy of the GNU General Public License
%along with this program; if not, write to the Free Software
%Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

% ver 1.2 100105 johan

% Check similarity matrix
S=sR.cluster.similarity;
if isempty(S),
  error('Similarities not computed: use icassoSimilarity');
end

% convert to dissimilarities using sim2dis (default operation: D=1-S)
D=sim2dis(sR.cluster.similarity);

% Init some variables
M=icassoGet(sR,'M');
dim=size(sR.signal,1);

if nargin<2|isempty(strategy),
  strategy='AL';
end
% we are case insensitive
stratgy=upper(strategy);
sR.cluster.strategy=strategy;

if nargin<3|isempty(L),
  L='auto';
end

if isnumeric(L),
  % The user has specified max number for clusters
  
  % Check L 
  if fix(L)~=L,
    error('L must be integer.');
  elseif L<2,
    error('L must be at least 2.');
  elseif L>M,
    error('L cannot be more than the number of estimates.');
  end
  if L>100,
    warning(['R-index requested for more that 100 clusters: this can' ...
	     ' be heavy.']);
  end
else
  if ~strcmp(lower(L),'auto'),
    error('Integer value or ''auto'' expected for the third input argument.');
  end
  % some heuristics for limiting the computation of clustering
  % validity indice: determines L reasonably L=ceil(3*original data dimension)
  %however, L is no more than there are estimates and in maximum 100 (too slow for bigger L).   
  
  L=ceil(3*dim);
  if L>M;
    L=M;
  end
  if L>100,
    L=100;
  end
end

% Change Icasso name to SOM Toolbox name
switch strategy
 case 'AL'
  strategy='average';
 case 'CL'
  strategy='complete';
 case 'SL'
  strategy='single';
 otherwise
  error(['Agglomeration strategy ' strategy ' not implemented.']);
end

%%%% Make partition %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
[sR.cluster.partition,sR.cluster.dendrogram.Z,sR.cluster.dendrogram.order]=...
    hcluster(D,strategy);

%%%%% Compute cluster validity %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% init R
sR.cluster.index.R=ones(M,1)*NaN;
sR.cluster.index.R(1:L,1)=rindex(D,sR.cluster.partition(1:L,:));  

  
