function sR=icassoSimilarity(sR,simfcn)
%function sR=icassoSimilarity(sR,[simfcn])
%
%PURPOSE
%
%To compute a similarity matrix, typically consisting of absolute
%values of correlation between the independent component estimates.
%
%EXAMPLES OF BASIC USAGE
%
%    sR=icassoSimilarity(sR); 
%
%uses the default way of computing the similarity matrix S between
%the estimates, i.e.,  the absolute value of their mutual correlation
%coefficient. The result is stored into Icasso result struct sR.
%
%   sR=icassoSimilarity(sR,'power')
%
%as previous, but similarity function 'power' is used instead.
%
%INPUT
%
%[An argument in brackets is optional. If it isn't  given or it's
% an empty matrix/string, the function will use a default value.] 
%
% sR       (struct) Icasso result data structure
% [simfcn] (string) 'abscorr' (default) | 'power' | 'corr'
%
%OUTPUT
%
% sR (struct) updated Icasso result data structure 
%
%The function updates _only_ the following fields:
%sR.cluster.simfcn, sR.cluster.similarity. See icassoStruct. 
%
%DETAILS
%
%simfcn     Similarity matrix S computed as follows:
%
%'abscorr'  S(i,j) the absolute value of Pearson's linear correlation
%            coefficient between estimates i and j.
%'power'    S(i,j) as 'abscorr' but the correlations are computed
%            between squared source estimates (IC estimates).
%'corr'     S(i,j) is Pearson's linear correlation coefficient between
%            estimates i and j.
%
%SEE ALSO
% icassoStruct
% icassoExp
% corrw

%COPYRIGHT NOTICE
%This function is a part of Icasso software library
%Copyright (C) 2003-2005 Johan Himberg
%
%This program is free software; you can redistribute it and/or
%modify it under the terms of the GNU General Public License
%as published by the Free Software Foundation; either version 2
%of the License, or any later version.
%
%This program is distributed in the hope that it will be useful,
%but WITHOUT ANY WARRANTY; without even the implied warranty of
%MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
%GNU General Public License for more details.
%
%You should have received a copy of the GNU General Public License
%along with this program; if not, write to the Free Software
%Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

% ver 1.2 johan 100105

%% Default simfcn
if nargin<2|isempty(simfcn),
  simfcn='abscorr';
end

% we are case insensitive
simfcn=lower(simfcn);

switch simfcn
 case 'corr'
  S=corrw(icassoGet(sR,'W'),icassoGet(sR,'dewhitemat'));
 case 'abscorr'
  S=abs(corrw(icassoGet(sR,'W'),icassoGet(sR,'dewhitemat')));
 case 'power'
  s=icassoGet(sR,'source');
  s=remmean(s).^2;
  S=abs(corrcoef(s'));
 otherwise
  error(['Unknown similarity function ''' simfcn '''.']);
end

% Just to make things sure

switch simfcn 
 case {'abscorr','power'}
  S(S>1)=1; S(S<0)=0;
 case 'corr'
  S(S>1)=1; S(S<-1)=-1;
end

% Store result & param.
sR.cluster.similarity=S;
sR.cluster.simfcn=simfcn;
